# coding=utf-8
# Copyright 2021 The Eleuther AI and HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
""" PyTorch GPT Neo model. """

import os
from typing import Tuple

import torch
import torch.nn.functional as F
import torch.utils.checkpoint
from torch import nn
from torch.nn import CrossEntropyLoss

from ...activations import ACT2FN
from ...file_utils import add_code_sample_docstrings, add_start_docstrings, \
    add_start_docstrings_to_model_forward
from ...modeling_outputs import (
    BaseModelOutputWithPast,
    BaseModelOutputWithPastAndCrossAttentions,
    CausalLMOutputWithCrossAttentions,
    CausalLMOutputWithPast,
)
from ...modeling_utils import PreTrainedModel
from ...utils import logging
from .configuration_gpt_neo import GPTNeoConfig
import math
from typing import Any, Optional

from einops import rearrange, repeat

logger = logging.get_logger(__name__)

_CONFIG_FOR_DOC = "GPTNeoConfig"
_TOKENIZER_FOR_DOC = "GPT2Tokenizer"

GPT_NEO_PRETRAINED_MODEL_ARCHIVE_LIST = [
    "EleutherAI/gpt-neo-1.3B",
    # See all GPTNeo models at https://huggingface.co/models?filter=gpt_neo
]

_CHECKPOINT_FOR_DOC = "EleutherAI/gpt-neo-1.3B"


def load_tf_weights_in_gpt_neo(model, config, gpt_neo_checkpoint_path):
    """Load tf checkpoints in a pytorch model"""
    try:
        import re

        import tensorflow as tf
    except ImportError:
        logger.error(
            "Loading a TensorFlow model in PyTorch, requires TensorFlow to be installed. Please see "
            "https://www.tensorflow.org/install/ for installation instructions."
        )
        raise
    tf_path = os.path.abspath(gpt_neo_checkpoint_path)
    logger.info(f"Converting TensorFlow checkpoint from {tf_path}")
    # Load weights from TF model
    init_vars = tf.train.list_variables(tf_path)
    names = []
    arrays = []
    for name, shape in init_vars:
        if "global_step" not in name and "adam" not in name:
            array = tf.train.load_variable(tf_path, name)
            array = tf.dtypes.cast(array.squeeze(), tf.float32).numpy()
            name = name.replace("attn/q", "attn/attention/q_proj/w")
            name = name.replace("attn/k", "attn/attention/k_proj/w")
            name = name.replace("attn/v", "attn/attention/v_proj/w")
            name = name.replace("attn/o", "attn/attention/out_proj/w")
            name = name.replace("norm_1", "ln_1")
            name = name.replace("norm_2", "ln_2")
            name = name.replace("attn/compute_output_bias/o_b",
                                "attn/attention/out_proj/b")
            name = name.replace("conv1d_main/c_fc/kernel", "c_fc/w")
            name = name.replace("conv1d_main/c_fc/bias", "c_fc/b")
            name = name.replace("conv1d_main/c_proj/kernel", "c_proj/w")
            name = name.replace("conv1d_main/c_proj/bias", "c_proj/b")

            names.append(name)
            arrays.append(array)

    for name, array in zip(names, arrays):
        name = name[5:]  # skip "gpt2/"
        name = name.split("/")
        pointer = model.transformer
        for m_name in name:
            if re.fullmatch(r"[A-Za-z]+\d+", m_name):
                scope_names = re.split(r"(\d+)", m_name)
            else:
                scope_names = [m_name]
            if scope_names[0] == "w" or scope_names[0] == "g":
                pointer = getattr(pointer, "weight")
            elif scope_names[0] == "b":
                pointer = getattr(pointer, "bias")
            elif scope_names[0] == "wpe" or scope_names[0] == "wte":
                pointer = getattr(pointer, scope_names[0])
                pointer = getattr(pointer, "weight")
            else:
                pointer = getattr(pointer, scope_names[0])
            if len(scope_names) >= 2:
                num = int(scope_names[1])
                pointer = pointer[num]

        if name[-1] == "w" and name[-2] in ["out_proj", "k_proj", "q_proj",
                                            "v_proj", "c_proj", "c_fc"]:
            array = array.transpose()

        if name == ["wte"]:
            # if vocab is padded, then trim off the padding embeddings
            array = array[: config.vocab_size]

        try:
            assert (
                    pointer.shape == array.shape
            ), f"Pointer shape {pointer.shape} and array shape {array.shape} mismatched {name}"
        except AssertionError as e:
            e.args += (pointer.shape, array.shape)
            raise
        print(f"Initialize PyTorch weight {name}")
        pointer.data = torch.from_numpy(array)

    # init the final linear layer using word embeddings
    embs = model.transformer.wte.weight
    lin = nn.Linear(embs.size()[1], embs.size()[0], bias=False)
    lin.weight = embs
    model.set_output_embeddings(lin)
    return model


def to_gpu(x, config, keep_dtype=False):
    #if hasattr(x, 'is_cuda') and x.is_cuda:
    #    return x
    model_dtypes = {"fp16": torch.float16, "fp32": torch.float32,
                    "bf16": torch.bfloat16}
    if not keep_dtype:
        x = x.to(model_dtypes[config.model_dtype])
    if config.model_device is not None:
        return x.to(config.model_device)
    colab = False
    space = False
    gb = 8000
    if config.jax:
        gb = 14500
    try:
        if 'google.colab' in str(get_ipython()):
            colab = True
    except:
        pass
    try:
        if torch.cuda.get_device_properties(0).total_memory > gb * 1024 * 1024:
            space = True
    except:
        pass
    if colab or space:
        return x.cuda()
    else:
        return x


def fixed_pos_embedding(dim=None, seq_len=None, x=None):
    if x is None:
        x = torch.tensor([])
    inv_freq = 1. / (10000 ** (torch.arange(0, dim, 2) / dim)).to(x.dtype).to(x.device)
    sinusoid_inp = torch.einsum('i , j -> i j', torch.arange(seq_len).to(x.device), inv_freq).float()

    return torch.sin(sinusoid_inp), torch.cos(sinusoid_inp)


def rotate_every_two(x):
    x1 = x[:, :, :, ::2]
    x2 = x[:, :, :, 1::2]
    x = torch.stack((-x2, x1), axis=-1)
    return rearrange(x, '... d j -> ... (d j)')


def apply_rotary_pos_emb(x, sincos, offset=0):
    sin, cos = map(lambda t: repeat(t[offset:x.shape[1] + offset, :],
                                    "n d -> () n () (d j)", j=2), sincos)
    return (x * cos) + (rotate_every_two(x) * sin)


def rotate_half(x):
    x1, x2 = x[..., :x.shape[-1] // 2], x[..., x.shape[-1] // 2:]
    return torch.cat((-x2, x1), dim=x1.ndim - 1)


@torch.jit.script
def apply_rotary_pos_emb_half_jit(x, sin, cos, offset: int = 0):
    sin = sin[None, offset:x.shape[1] + offset, None, :]
    cos = cos[None, offset:x.shape[1] + offset, None, :]
    sin = torch.cat((sin, sin), dim=-1)
    cos = torch.cat((cos, cos), dim=-1)
    return (x * cos) + (rotate_half(x) * sin)


def apply_rotary_pos_emb_half(x, sincos, offset=0):
    if x.dtype == torch.float16:
        return apply_rotary_pos_emb_half_jit(x, sincos[0], sincos[1], offset)
    sin, cos = map(lambda t: repeat(t[offset:x.shape[1] + offset, :],
                                    "n d -> () n () (j d)", j=2), sincos)
    return (x * cos) + (rotate_half(x) * sin)


def make_positions(tensor, padding_idx: int, onnx_trace: bool = False):
    """Replace non-padding symbols with their position numbers.

    Position numbers begin at padding_idx+1. Padding symbols are ignored.
    """
    # The series of casts and type-conversions here are carefully
    # balanced to both work with ONNX export and XLA. In particular XLA
    # prefers ints, cumsum defaults to output longs, and ONNX doesn't know
    # how to handle the dtype kwarg in cumsum.
    mask = tensor.ne(torch.tensor(50257, requires_grad=False)).int()
    return (torch.cumsum(mask, dim=1).type_as(mask) * mask).long() + padding_idx


class SinusoidalPositionalEmbedding(nn.Module):
    """This module produces sinusoidal positional embeddings of any length.

    Padding symbols are ignored.
    """

    def __init__(self, embedding_dim, padding_idx, init_size=1024):
        super().__init__()
        self.embedding_dim = embedding_dim
        self.padding_idx = padding_idx if padding_idx is not None else 0
        self.weights = SinusoidalPositionalEmbedding.get_embedding(
            init_size, embedding_dim, padding_idx
        )
        self.onnx_trace = False
        self.register_buffer("_float_tensor",
                             torch.tensor(1.0, requires_grad=False).float())
        self.max_positions = int(1e5)
        # print(embedding_dim, padding_idx, init_size)

    def prepare_for_onnx_export_(self):
        self.onnx_trace = True

    @staticmethod
    def get_embedding(
            num_embeddings: int, embedding_dim: int,
            padding_idx: Optional[int] = None
    ):
        """Build sinusoidal embeddings.

        This matches the implementation in tensor2tensor, but differs slightly
        from the description in Section 3.5 of "Attention Is All You Need".
        """
        half_dim = embedding_dim // 2
        emb = math.log(10000) / (half_dim - 1)
        emb = torch.exp(torch.arange(half_dim, dtype=torch.float) * -emb)
        emb = torch.arange(num_embeddings, dtype=torch.float).unsqueeze(
            1
        ) * emb.unsqueeze(0)
        emb = torch.cat([torch.sin(emb), torch.cos(emb)], dim=1).view(
            num_embeddings, -1
        )
        if embedding_dim % 2 == 1:
            # zero pad
            emb = torch.cat([emb, torch.zeros(num_embeddings, 1)], dim=1)
        if padding_idx is not None:
            emb[padding_idx, :] = 0
        return emb

    def forward(
            self,
            input,
            incremental_state: Optional[Any] = None,
            timestep: Optional[torch.Tensor] = None,
            positions: Optional[Any] = None,
            offset: Optional[int] = 0
    ):
        """Input is expected to be of size [bsz x seqlen]."""
        bspair = input.shape
        bsz, seq_len = bspair[0], bspair[1]
        max_pos = self.padding_idx + 1 + seq_len + offset
        # print("max_pos: " + str(max_pos))
        if self.weights is None or max_pos > self.weights.size(0):
            # print("recomputing embeddings")
            # recompute/expand embeddings if needed
            self.weights = SinusoidalPositionalEmbedding.get_embedding(
                max_pos, self.embedding_dim, self.padding_idx + offset
            )
        self.weights = self.weights.to(self._float_tensor)

        if incremental_state is not None:
            # positions is the same for every token when decoding a single step
            pos = timestep.view(-1)[0] + 1 if timestep is not None else seq_len
            if self.onnx_trace:
                return (
                    self.weights.index_select(
                        index=self.padding_idx + pos + offset, dim=0)
                        .unsqueeze(1)
                        .repeat(bsz, 1, 1)
                )
            return self.weights[self.padding_idx + pos + offset, :].expand(bsz,
                                                                           1,
                                                                           -1)

        positions = make_positions(
            input, self.padding_idx + offset, onnx_trace=self.onnx_trace
        )
        if self.onnx_trace:
            flat_embeddings = self.weights.detach().index_select(0,
                                                                 positions.view(
                                                                     -1))
            embedding_shape = torch.cat(
                (bsz.view(1), seq_len.view(1),
                 torch.tensor([-1], dtype=torch.long))
            )
            embeddings = torch.onnx.operators.reshape_from_tensor_shape(
                flat_embeddings, embedding_shape
            )
            return embeddings
        return (
            self.weights.index_select(0, positions.view(-1))
                .view(bsz, seq_len, -1)
                .detach()
        )


def PositionalEmbedding(
        num_embeddings: int,
        embedding_dim: int,
        padding_idx: int,
):
    m = SinusoidalPositionalEmbedding(
        embedding_dim,
        padding_idx,
        init_size=num_embeddings + padding_idx + 1,
    )
    return m


class GPTNeoAttentionMixin:
    """
    A few attention related utilities for attention modules in GPT Neo, to be used as a mixin.
    """

    def _split_heads(self, tensor, num_heads, attn_head_size, rotary):
        """
        Splits hidden_size dim into attn_head_size and num_heads
        """
        new_shape = tensor.size()[:-1] + (num_heads, attn_head_size)
        tensor = tensor.view(*new_shape)
        if rotary:
            return tensor
        if len(tensor.shape) == 5:
            return tensor.permute(0, 1, 3, 2,
                                  4)  # (batch, blocks, head, block_length, head_features)
        elif len(tensor.shape) == 4:
            return tensor.permute(0, 2, 1,
                                  3)  # (batch, head, seq_length, head_features)
        else:
            raise ValueError(
                f"Input tensor rank should be one of [4, 5], but is: {len(tensor.shape)}")

    def _merge_heads(self, tensor, num_heads, attn_head_size):
        """
        Merges attn_head_size dim and num_attn_heads dim into hidden_size
        """
        if len(tensor.shape) == 5:
            tensor = tensor.permute(0, 1, 3, 2, 4).contiguous()
        elif len(tensor.shape) == 4:
            tensor = tensor.permute(0, 2, 1, 3).contiguous()
        else:
            raise ValueError(
                f"Input tensor rank should be one of [4, 5], but is: {len(tensor.shape)}")
        new_shape = tensor.size()[:-2] + (num_heads * attn_head_size,)
        return tensor.view(new_shape)

    def _attn(self,
              query: torch.Tensor,
              key: torch.Tensor,
              value: torch.Tensor,
              causal_mask: torch.Tensor,
              masked_bias: torch.Tensor,
              attn_dropout: torch.nn.Dropout,
              attention_mask: [None, torch.Tensor] = None,
              head_mask: [None, torch.Tensor] = None,
              scale_attn: [None, torch.Tensor] = None,
              full_bf16=False, rotary_half=False, force_fp32_attn=True):
        if force_fp32_attn or (
                rotary_half or not full_bf16 and query.dtype is torch.bfloat16):
            attn_weights = torch.matmul(query.float(),
                                        key.transpose(-1, -2).float())
        else:
            attn_weights = torch.matmul(query, key.transpose(-1, -2))
        if masked_bias.dtype != attn_weights.dtype:
            masked_bias = masked_bias.to(attn_weights.dtype)
        attn_weights = torch.where(causal_mask, attn_weights, masked_bias)

        if scale_attn is not None:
            attn_weights = attn_weights / scale_attn

        if attention_mask is not None:
            # Apply the attention mask
            attn_weights = attn_weights + attention_mask

        attn_weights = nn.Softmax(dim=-1)(attn_weights)
        attn_weights = attn_weights.to(value.dtype)
        attn_weights = attn_dropout(attn_weights)

        # Mask heads if we want to
        if head_mask is not None:
            attn_weights = attn_weights * head_mask

        if not full_bf16 and value.dtype is torch.bfloat16:
            attn_output = torch.matmul(attn_weights.float(), value.float()).to(
                value.dtype)
        else:
            attn_output = torch.matmul(attn_weights, value).to(value.dtype)

        return attn_output, attn_weights


class GPTNeoSelfAttention(nn.Module, GPTNeoAttentionMixin):
    def __init__(self, attention_type, config):
        super().__init__()

        self.window_size = None
        max_positions = config.max_position_embeddings
        bias = torch.tril(
            torch.ones((max_positions, max_positions), dtype=torch.uint8,
                       requires_grad=False)).view(
            1, 1, max_positions, max_positions
        ).bool()

        if attention_type == "local":
            self.register_buffer(
                "bias",
                bias ^ torch.tril(bias, -config.window_size),
            )
        else:
            self.register_buffer(
                "bias",
                bias,
            )

        if config.rotary_half:
            self.register_buffer("masked_bias",
                                 torch.tensor(-10000, requires_grad=False))
        else:
            self.register_buffer("masked_bias",
                                 torch.tensor(-1e9, requires_grad=False))

        self.attn_dropout = nn.Dropout(config.attention_dropout)
        self.resid_dropout = nn.Dropout(config.resid_dropout)

        self.embed_dim = config.hidden_size
        self.num_heads = config.num_heads
        self.head_dim = self.embed_dim // self.num_heads
        if config.jax or config.fair:
            self.register_buffer("scale_attn", torch.sqrt(
                torch.tensor(self.head_dim, requires_grad=False).float()))
        else:
            self.scale_attn = None
        if self.head_dim * self.num_heads != self.embed_dim:
            raise ValueError(
                f"embed_dim must be divisible by num_heads (got `embed_dim`: {self.embed_dim} and `num_heads`: {self.num_heads})."
            )

        self.fair = config.fair
        if self.fair or config.rotary_half:
            attn_bias = True
        else:
            attn_bias = False
        self.rotary_half = config.rotary_half
        self.k_proj = nn.Linear(self.embed_dim, self.embed_dim, bias=attn_bias)
        self.v_proj = nn.Linear(self.embed_dim, self.embed_dim, bias=attn_bias)
        self.q_proj = nn.Linear(self.embed_dim, self.embed_dim, bias=attn_bias)
        self.out_proj = nn.Linear(self.embed_dim, self.embed_dim,
                                  bias=not config.jax or config.fair or config.rotary_half)
        self.full_bf16 = config.full_bf16
        if hasattr(config, 'force_fp32_attn'):
            self.force_fp32_attn = config.force_fp32_attn
        else:
            self.force_fp32_attn = False
        self.rotary = config.rotary
        self.rotary_dim = self.head_dim
        self.rotary_func = apply_rotary_pos_emb
        if self.rotary_half:
            self.rotary_func = apply_rotary_pos_emb_half
        if config.rotary_dim is not None:
            self.rotary_dim = config.rotary_dim
        if self.rotary:
            if self.rotary_half:
                sin, cos = fixed_pos_embedding(dim=self.rotary_dim,
                                               seq_len=max_positions,
                                               x=to_gpu(torch.tensor([]), config))
            else:
                sin, cos = fixed_pos_embedding(dim=self.rotary_dim,
                                               seq_len=max_positions)
            self.register_buffer("sin", sin)
            self.register_buffer("cos", cos)

    def forward(
            self,
            hidden_states,
            attention_mask=None,
            layer_past=None,
            head_mask=None,
            use_cache=False,
            output_attentions=False,
    ):

        query = self.q_proj(hidden_states)
        key = self.k_proj(hidden_states)
        value = self.v_proj(hidden_states)

        query = self._split_heads(query, self.num_heads, self.head_dim,
                                  self.rotary)
        key = self._split_heads(key, self.num_heads, self.head_dim, self.rotary)
        value = self._split_heads(value, self.num_heads, self.head_dim, False)

        if self.rotary:
            seq_len = key.shape[1]
            offset = 0
            if layer_past is not None:
                offset = layer_past[0].shape[-2]
                seq_len += offset
            if self.rotary_dim < self.head_dim:
                k_rot = key[:, :, :, :self.rotary_dim]
                k_pass = key[:, :, :, self.rotary_dim:]

                q_rot = query[:, :, :, :self.rotary_dim]
                q_pass = query[:, :, :, self.rotary_dim:]

                k_rot = self.rotary_func(k_rot, (self.sin, self.cos),
                                         offset=offset).to(k_rot.dtype)
                q_rot = self.rotary_func(q_rot, (self.sin, self.cos),
                                         offset=offset).to(q_rot.dtype)

                key = torch.cat([k_rot, k_pass], dim=-1)
                query = torch.cat([q_rot, q_pass], dim=-1)
            elif self.rotary:
                key = self.rotary_func(key, (self.sin, self.cos),
                                       offset=offset).to(key.dtype)
                query = self.rotary_func(query, (self.sin, self.cos),
                                         offset=offset).to(query.dtype)
            key = key.permute(0, 2, 1, 3)
            query = query.permute(0, 2, 1, 3)

        if layer_past is not None:
            past_key = layer_past[0]
            past_value = layer_past[1]
            key = torch.cat((past_key, key), dim=-2).to(key.dtype)
            value = torch.cat((past_value, value), dim=-2).to(value.dtype)

        if use_cache is True:
            present = (key, value)
        else:
            present = None

        query_length, key_length = query.size(-2), key.size(-2)
        causal_mask = self.bias[:, :, key_length - query_length: key_length,
                      :key_length]

        if not hasattr(self, 'force_fp32_attn'):
            self.force_fp32_attn = False

        attn_output, attn_weights = self._attn(
            query, key, value, causal_mask, self.masked_bias, self.attn_dropout,
            attention_mask, head_mask, self.scale_attn, self.full_bf16,
            self.rotary_half, self.force_fp32_attn
        )

        attn_output = self._merge_heads(attn_output, self.num_heads,
                                        self.head_dim)
        attn_output = self.out_proj(attn_output)
        attn_output = self.resid_dropout(attn_output)

        outputs = (attn_output, present)
        if output_attentions:
            outputs += (attn_weights,)

        return outputs  # a, present, (attentions)


class GPTNeoAttention(nn.Module):
    def __init__(self, config, layer_id=0):
        super().__init__()
        self.layer_id = layer_id
        self.attention_layers = config.attention_layers
        self.attention_type = self.attention_layers[layer_id]

        if self.attention_type in ["global", "local"]:
            self.attention = GPTNeoSelfAttention(self.attention_type, config)
        else:
            raise NotImplementedError(
                "Only attn layer types 'global' and 'local' exist, but got `config.attention_layers`: "
                f"{config.attention_layers}. Select attn layer types from ['global', 'local'] only."
            )

    def forward(
            self,
            hidden_states,
            layer_past=None,
            attention_mask=None,
            head_mask=None,
            use_cache=False,
            output_attentions=False,
    ):
        outputs = self.attention(
            hidden_states,
            attention_mask=attention_mask,
            layer_past=layer_past,
            head_mask=head_mask,
            use_cache=use_cache,
            output_attentions=output_attentions,
        )

        return outputs


@torch.jit.script
def bias_gelu(bias, y):
    x = bias + y
    return x * 0.5 * (1.0 + torch.tanh(0.79788456 * x * (1 + 0.044715 * x * x)))


class GPTNeoMLP(nn.Module):
    def __init__(self, intermediate_size,
                 config):  # in MLP: intermediate_size= 4 * hidden_size
        super().__init__()
        embed_dim = config.hidden_size
        self.c_fc = nn.Linear(embed_dim, intermediate_size)
        self.c_proj = nn.Linear(intermediate_size, embed_dim)
        self.act = ACT2FN[config.activation_function]
        self.dropout = nn.Dropout(config.resid_dropout)
        self.rotary_half = config.rotary_half
        self.modified_bias = False

    def forward(self, hidden_states):
        if self.rotary_half and not self.modified_bias:
            try:
                if self.bias_fc is not None:
                    self.modified_bias = True
            except:
                pass
            if not self.modified_bias:
                bias = self.c_fc.bias
                self.c_fc.bias = None
                self.register_buffer("bias_fc", bias)
                self.modified_bias = True
        hidden_states = self.c_fc(hidden_states)
        if self.rotary_half:
            hidden_states = bias_gelu(self.bias_fc, hidden_states)
        else:
            hidden_states = self.act(hidden_states)
        hidden_states = self.c_proj(hidden_states)
        hidden_states = self.dropout(hidden_states)
        return hidden_states


class GPTNeoBlock(nn.Module):
    def __init__(self, config, layer_id):
        super().__init__()
        hidden_size = config.hidden_size
        inner_dim = config.intermediate_size if config.intermediate_size is not None else 4 * hidden_size
        self.ln_1 = nn.LayerNorm(hidden_size, eps=config.layer_norm_epsilon)
        self.attn = GPTNeoAttention(config, layer_id)
        self.jax = config.jax
        self.rotary_half = config.rotary_half
        if not self.jax or self.rotary_half:
            self.ln_2 = nn.LayerNorm(hidden_size, eps=config.layer_norm_epsilon)
        self.mlp = GPTNeoMLP(inner_dim, config)

        self.layer_id = layer_id
        self.pipe_device = None

    def forward(
            self,
            hidden_states,
            layer_past=None,
            attention_mask=None,
            head_mask=None,
            use_cache=False,
            output_attentions=False,
            hypernetwork=None,
            every_n=5,
    ):
        residual = hidden_states
        hidden_states = self.ln_1(hidden_states)
        attn_outputs = self.attn(
            hidden_states,
            layer_past=layer_past,
            attention_mask=attention_mask,
            head_mask=head_mask,
            use_cache=use_cache,
            output_attentions=output_attentions,
        )
        attn_output = attn_outputs[0]  # output_attn: a, present, (attentions)
        outputs = attn_outputs[1:]

        if hypernetwork and self.layer_id % every_n == 0:
            hyper_out = hypernetwork(hidden_states)


        if self.jax:
            if self.rotary_half:
                feed_forward_hidden_states = self.mlp(self.ln_2(residual))
            else:
                feed_forward_hidden_states = self.mlp(hidden_states)
            hidden_states = attn_output + feed_forward_hidden_states + residual
        else:
            # residual connection
            hidden_states = attn_output + residual

            residual = hidden_states
            hidden_states = self.ln_2(hidden_states)
            feed_forward_hidden_states = self.mlp(hidden_states)
            # residual connection
            hidden_states = residual + feed_forward_hidden_states

        if hypernetwork and self.layer_id % every_n == 0:
            hidden_states = hidden_states + hyper_out

        if use_cache:
            outputs = (hidden_states,) + outputs
        else:
            outputs = (hidden_states,) + outputs[1:]

        return outputs  # hidden_states, present, (attentions, cross_attentions)


class GPTNeoBlockPipe(nn.Module):
    def __init__(self, block, dev):
        super().__init__()
        self.block = block.to(dev)
        self.device = dev

    def forward(self, *x):
        if len(x) > 1:
            past = None
            more_past = torch.tensor([0] * x[0].shape[0]).to(self.device)
            if len(x) > 1 and len(x[1].shape) > 1:
                past = x[1][:, 0].transpose(0, 1)
                more_past = x[1][:, 1:]
            outputs = self.block(
                x[0],
                layer_past=past,
                attention_mask=None,
                head_mask=None,
                use_cache=True,
                output_attentions=False,
            )
            presents = torch.stack(outputs[1], dim=0).transpose(0, 1).unsqueeze(
                1)
            if self.block.layer_id == 0:
                return (outputs[0], more_past, presents)
            else:
                presents = torch.cat([x[2], presents], dim=1)
                return (outputs[0], more_past, presents)
        else:
            outputs = self.block(
                x[0],
                layer_past=None,
                attention_mask=None,
                head_mask=None,
                use_cache=False,
                output_attentions=False,
            )
            return outputs[0]


class PipePassTuple(nn.Module):
    def __init__(self, module, dev):
        super().__init__()
        self.module = module.to(dev)
        self.device = dev

    def forward(self, *x):
        if len(x) > 1:
            return (self.module(x[0]),) + x[1:]
        else:
            return self.module(x[0])


class GPTNeoPreTrainedModel(PreTrainedModel):
    """
    An abstract class to handle weights initialization and a simple interface for downloading and loading pretrained
    models.
    """

    config_class = GPTNeoConfig
    load_tf_weights = load_tf_weights_in_gpt_neo
    base_model_prefix = "transformer"

    def __init__(self, *inputs, **kwargs):
        super().__init__(*inputs, **kwargs)

    def _init_weights(self, module):
        """Initialize the weights."""
        if isinstance(module, (nn.Linear,)):
            # Slightly different from the TF version which uses truncated_normal for initialization
            # cf https://github.com/pytorch/pytorch/pull/5617
            module.weight.data.normal_(mean=0.0,
                                       std=self.config.initializer_range)
            if module.bias is not None:
                module.bias.data.zero_()
        elif isinstance(module, nn.Embedding):
            module.weight.data.normal_(mean=0.0,
                                       std=self.config.initializer_range)
            if module.padding_idx is not None:
                module.weight.data[module.padding_idx].zero_()
        elif isinstance(module, nn.LayerNorm):
            module.bias.data.zero_()
            module.weight.data.fill_(1.0)


GPT_NEO_START_DOCSTRING = r"""

    This model inherits from :class:`~transformers.PreTrainedModel`. Check the superclass documentation for the generic
    methods the library implements for all its model (such as downloading or saving, resizing the input embeddings,
    pruning heads etc.)

    This model is also a PyTorch `torch.nn.Module <https://pytorch.org/docs/stable/nn.html#torch.nn.Module>`__
    subclass. Use it as a regular PyTorch Module and refer to the PyTorch documentation for all matter related to
    general usage and behavior.

    Parameters:
        config (:class:`~transformers.GPTNeoConfig`): Model configuration class with all the parameters of the model.
            Initializing with a config file does not load the weights associated with the model, only the
            configuration. Check out the :meth:`~transformers.PreTrainedModel.from_pretrained` method to load the model
            weights.
"""

GPT_NEO_INPUTS_DOCSTRING = r"""
    Args:
        input_ids (:obj:`torch.LongTensor` of shape :obj:`(batch_size, input_ids_length)`):
            :obj:`input_ids_length` = ``sequence_length`` if :obj:`past_key_values` is ``None`` else
            ``past_key_values[0][0].shape[-2]`` (``sequence_length`` of input past key value states). Indices of input
            sequence tokens in the vocabulary.

            If :obj:`past_key_values` is used, only ``input_ids`` that do not have their past calculated should be
            passed as ``input_ids``.

            Indices can be obtained using :class:`~transformers.GPTNeoTokenizer`. See
            :meth:`transformers.PreTrainedTokenizer.encode` and :meth:`transformers.PreTrainedTokenizer.__call__` for
            details.

            `What are input IDs? <../glossary.html#input-ids>`__
        past_key_values (:obj:`Tuple[Tuple[torch.Tensor]]` of length :obj:`config.num_layers`):
            Contains precomputed hidden-states (key and values in the attention blocks) as computed by the model (see
            :obj:`past_key_values` output below). Can be used to speed up sequential decoding. The ``input_ids`` which
            have their past given to this model should not be passed as ``input_ids`` as they have already been
            computed.
        attention_mask (:obj:`torch.FloatTensor` of shape :obj:`(batch_size, sequence_length)`, `optional`):
            Mask to avoid performing attention on padding token indices. Mask values selected in ``[0, 1]``:

            - 1 for tokens that are **not masked**,
            - 0 for tokens that are **masked**.

            `What are attention masks? <../glossary.html#attention-mask>`__
        token_type_ids (:obj:`torch.LongTensor` of shape :obj:`(batch_size, input_ids_length)`, `optional`):
            Segment token indices to indicate first and second portions of the inputs. Indices are selected in ``[0,
            1]``:

            - 0 corresponds to a `sentence A` token,
            - 1 corresponds to a `sentence B` token.

            `What are token type IDs? <../glossary.html#token-type-ids>`_
        position_ids (:obj:`torch.LongTensor` of shape :obj:`(batch_size, sequence_length)`, `optional`):
            Indices of positions of each input sequence tokens in the position embeddings. Selected in the range ``[0,
            config.max_position_embeddings - 1]``.

            `What are position IDs? <../glossary.html#position-ids>`_
        head_mask (:obj:`torch.FloatTensor` of shape :obj:`(num_heads,)` or :obj:`(num_layers, num_heads)`, `optional`):
            Mask to nullify selected heads of the self-attention modules. Mask values selected in ``[0, 1]``:

            - 1 indicates the head is **not masked**,
            - 0 indicates the head is **masked**.

        inputs_embeds (:obj:`torch.FloatTensor` of shape :obj:`(batch_size, sequence_length, hidden_size)`, `optional`):
            Optionally, instead of passing :obj:`input_ids` you can choose to directly pass an embedded representation.
            This is useful if you want more control over how to convert :obj:`input_ids` indices into associated
            vectors than the model's internal embedding lookup matrix.

            If :obj:`past_key_values` is used, optionally only the last :obj:`inputs_embeds` have to be input (see
            :obj:`past_key_values`).
        use_cache (:obj:`bool`, `optional`):
            If set to :obj:`True`, :obj:`past_key_values` key value states are returned and can be used to speed up
            decoding (see :obj:`past_key_values`).
        output_attentions (:obj:`bool`, `optional`):
            Whether or not to return the attentions tensors of all attention layers. See ``attentions`` under returned
            tensors for more detail.
        output_hidden_states (:obj:`bool`, `optional`):
            Whether or not to return the hidden states of all layers. See ``hidden_states`` under returned tensors for
            more detail.
        return_dict (:obj:`bool`, `optional`):
            Whether or not to return a :class:`~transformers.file_utils.ModelOutput` instead of a plain tuple.
"""


@add_start_docstrings(
    "The bare GPT Neo Model transformer outputting raw hidden-states without any specific head on top.",
    GPT_NEO_START_DOCSTRING,
)
class GPTNeoModel(GPTNeoPreTrainedModel):
    def __init__(self, config):
        super().__init__(config)

        self.embed_dim = config.hidden_size
        if config.fair:
            self.register_buffer("embed_scale", to_gpu(torch.sqrt(
                torch.tensor(config.hidden_size, requires_grad=False)), config))
        self.jax = config.jax
        self.vocab_size = config.vocab_size
        self.wte = to_gpu(nn.Embedding(config.vocab_size, self.embed_dim),
                          config)
        if not config.rotary and not config.fair:
            # print("wpe")
            self.wpe = to_gpu(
                nn.Embedding(config.max_position_embeddings, self.embed_dim),
                config)

        if config.fair:
            # print("wpe_sin")
            self.wpe_sin = to_gpu(
                PositionalEmbedding(config.max_position_embeddings,
                                    self.embed_dim, 1),
                config)  # returns sinusoidal embeddings of shape: (1, 2049, 768)

        self.drop = to_gpu(nn.Dropout(config.embed_dropout), config)

        quantize = lambda x, i: x
        if config.eight_bit is not None:
            from .eight_bit_utils import dummify
            quantize = lambda x, i: dummify(x) if i in config.eight_bit else x

        self.h = nn.ModuleList(
            [quantize(to_gpu(GPTNeoBlock(config, layer_id=i), config), i) for i
             in range(config.num_layers)])
        self.ln_f = to_gpu(
            nn.LayerNorm(self.embed_dim, eps=config.layer_norm_epsilon), config)
        self.rotary = config.rotary
        self.fair = config.fair
        self.pipe = config.pipe
        self.need_pipe = config.need_pipe
        self.lm_head = None
        self.sin_scale = 1.0

        self.init_weights()

    def get_input_embeddings(self):
        return self.wte

    def set_input_embeddings(self, new_embeddings):
        self.wte = new_embeddings

    def setup_pipe(self):
        if self.pipe is None:
            from torch.distributed.pipeline.sync import Pipe, WithDevice
            modules = []
            partition = []
            for dev, layers in self.need_pipe["partitions"]:
                device = torch.device(dev)
                for layer in layers:
                    self.h[layer].pipe_device = device
                    partition.append(
                        GPTNeoBlockPipe(self.h[layer], device).to(device))
                modules.append(partition)
                partition = []
            del self.need_pipe["partitions"]
            device = modules[-1][-1].block.pipe_device
            modules[-1].append(PipePassTuple(self.ln_f, device).to(device))
            if self.lm_head is not None:
                modules[-1].append(
                    PipePassTuple(self.lm_head, device).to(device))
                self.lm_head = None
            modules = list(
                map(lambda partition: nn.Sequential(*partition), modules))
            self.pipe = nn.Sequential(*modules)
            self.pipe = Pipe(self.pipe, **self.need_pipe)

    @add_start_docstrings_to_model_forward(GPT_NEO_INPUTS_DOCSTRING)
    @add_code_sample_docstrings(
        tokenizer_class=_TOKENIZER_FOR_DOC,
        checkpoint=_CHECKPOINT_FOR_DOC,
        output_type=BaseModelOutputWithPastAndCrossAttentions,
        config_class=_CONFIG_FOR_DOC,
    )
    def forward(
            self,
            input_ids=None,
            past_key_values=None,
            attention_mask=None,
            token_type_ids=None,
            position_ids=None,
            head_mask=None,
            inputs_embeds=None,
            use_cache=None,
            output_attentions=None,
            output_hidden_states=None,
            return_dict=None,
            embs=None,
            hypernetwork=None,
    ):
        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )
        use_cache = use_cache if use_cache is not None else (
                    self.config.use_cache and self.pipe is None)
        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        if input_ids is not None and inputs_embeds is not None:
            raise ValueError(
                "You cannot specify both input_ids and inputs_embeds at the same time")
        elif input_ids is not None:
            input_shape = input_ids.size()
            input_ids = input_ids.view(-1, input_shape[-1])
            batch_size = input_ids.shape[0]
        elif inputs_embeds is not None:
            input_shape = inputs_embeds.size()[:-1]
            batch_size = inputs_embeds.shape[0]
        else:
            raise ValueError(
                "You have to specify either input_ids or inputs_embeds")

        device = input_ids.device if input_ids is not None else inputs_embeds.device

        if token_type_ids is not None:
            token_type_ids = token_type_ids.view(-1, input_shape[-1])
        if position_ids is not None:
            position_ids = position_ids.view(-1, input_shape[-1])

        if past_key_values is None:
            past_length = 0
            past_key_values = tuple([None] * len(self.h))
        else:
            past_length = past_key_values[0][0].size(-2)

        if position_ids is None:
            position_ids = torch.arange(past_length,
                                        input_shape[-1] + past_length,
                                        dtype=torch.long, device=device)
            position_ids = position_ids.unsqueeze(0).view(-1, input_shape[-1])

        # Attention mask.
        if attention_mask is not None:
            assert batch_size > 0, "batch_size has to be defined and > 0"
            global_attention_mask = attention_mask.view(batch_size, -1)
            # We create a 3D attention mask from a 2D tensor mask.
            # Sizes are [batch_size, 1, 1, to_seq_length]
            # So we can broadcast to [batch_size, num_heads, from_seq_length, to_seq_length]
            # this attention mask is more simple than the triangular masking of causal attention
            # used in OpenAI GPT, we just need to prepare the broadcast dimension here.
            global_attention_mask = global_attention_mask[:, None, None, :]

            # Since global_attention_mask is 1.0 for positions we want to attend and 0.0 for
            # masked positions, this operation will create a tensor which is 0.0 for
            # positions we want to attend and -10000.0 for masked positions.
            # Since we are adding it to the raw scores before the softmax, this is
            # effectively the same as removing these entirely.
            global_attention_mask = global_attention_mask.to(
                dtype=self.dtype)  # fp16 compatibility
            global_attention_mask = (1.0 - global_attention_mask) * -10000.0
        else:
            global_attention_mask = None

        # Local causal attention mask
        batch_size, seq_length = input_shape
        full_seq_length = seq_length + past_length

        # Prepare head mask if needed
        # 1.0 in head_mask indicate we keep the head
        # attention_probs has shape bsz x num_heads x N x N
        # head_mask has shape n_layer x batch x num_heads x N x N
        head_mask = self.get_head_mask(head_mask, self.config.num_layers)

        if inputs_embeds is None:
            inputs_embeds = self.wte(input_ids)

        if embs is not None and not (
                use_cache is not None and use_cache and past_key_values is not None and len(
                past_key_values) > 0 and past_key_values[0] is not None):
            offset = 0
            for pos, emb in embs:
                pos += offset
                if len(emb.shape) == 2:
                    emb = repeat(emb, "s d -> (b) s d", b=input_shape[0])
                inputs_embeds[:, pos:pos + emb.shape[1]] = emb
                offset += emb.shape[1]

        if self.rotary:
            hidden_states = inputs_embeds
        if self.fair:
            if not self.rotary:
                position_embeds = self.wpe_sin(input_ids,
                                               offset=past_length) * self.sin_scale
                inputs_embeds = inputs_embeds * self.embed_scale
                hidden_states = inputs_embeds + position_embeds
            else:
                inputs_embeds = inputs_embeds * self.embed_scale
                hidden_states = inputs_embeds
        if not (self.rotary or self.fair):
            position_embeds = self.wpe(position_ids)
            hidden_states = inputs_embeds + position_embeds

        if token_type_ids is not None:
            token_type_embeds = self.wte(token_type_ids)
            hidden_states = hidden_states + token_type_embeds

        hidden_states = self.drop(hidden_states)

        output_shape = input_shape + (hidden_states.size(-1),)

        presents = () if use_cache else None
        all_self_attentions = () if output_attentions else None
        all_hidden_states = () if output_hidden_states else None
        if self.need_pipe is not None:
            if use_cache:
                if isinstance(past_key_values, tuple):
                    hidden_states, past, presents = self.pipe(hidden_states,
                                                              torch.tensor([0] *
                                                                           hidden_states.shape[
                                                                               0]).to(
                                                                  hidden_states.device)).local_value()
                    del past
                else:
                    hidden_states, past, presents = self.pipe(hidden_states,
                                                              past_key_values).local_value()
                    del past
            else:
                hidden_states = self.pipe(hidden_states).local_value()
        else:
            for i, (block, layer_past) in enumerate(
                    zip(self.h, past_key_values)):
                attn_type = self.config.attention_layers[i]
                attn_mask = global_attention_mask

                if output_hidden_states:
                    all_hidden_states = all_hidden_states + (
                    hidden_states.cpu(),)

                if getattr(self.config, "gradient_checkpointing",
                           False) and self.training:

                    if use_cache:
                        logger.warning(
                            "`use_cache=True` is incompatible with `config.gradient_checkpointing=True`. Setting "
                            "`use_cache=False`..."
                        )
                        use_cache = False

                    def create_custom_forward(module):
                        def custom_forward(*inputs):
                            # None for past_key_value
                            return module(*inputs, use_cache, output_attentions)

                        return custom_forward

                    outputs = torch.utils.checkpoint.checkpoint(
                        create_custom_forward(block),
                        hidden_states,
                        None,
                        attn_mask,
                        head_mask[i],
                    )
                else:
                    outputs = block(
                        hidden_states,
                        layer_past=layer_past,
                        attention_mask=attn_mask,
                        head_mask=head_mask[i],
                        use_cache=use_cache,
                        output_attentions=output_attentions,
                        hypernetwork=hypernetwork,
                    )

                hidden_states = outputs[0]
                if use_cache is True:
                    presents = presents + (outputs[1],)

                if output_attentions:
                    all_self_attentions = all_self_attentions + (
                    outputs[2 if use_cache else 1],)

            hidden_states = self.ln_f(hidden_states)
            hidden_states = hidden_states.view(*output_shape)

        # Add last hidden state
        if output_hidden_states:
            all_hidden_states = all_hidden_states + (hidden_states,)

        if not return_dict:
            return tuple(v for v in [hidden_states, presents, all_hidden_states,
                                     all_self_attentions] if v is not None)

        return BaseModelOutputWithPast(
            last_hidden_state=hidden_states,
            past_key_values=presents,
            hidden_states=all_hidden_states,
            attentions=all_self_attentions,
        )


@add_start_docstrings(
    """
    The GPT Neo Model transformer with a language modeling head on top (linear layer with weights tied to the input
    embeddings).
    """,
    GPT_NEO_START_DOCSTRING,
)
class GPTNeoForCausalLM(GPTNeoPreTrainedModel):
    _keys_to_ignore_on_load_missing = [r"h\.\d+\.attn\.attention\.masked_bias",
                                       r"h\.\d+\.attn\.masked_bias",
                                       r"h\.\d+\.attn\.attention\.scale_attn",
                                       r"h\.\d+\.attn\.attention\.(sin|cos)",
                                       r"lm_head\.weight",
                                       r"h\.\d+\.attn\.attention\.bias",
                                       r"embed_scale"]

    # _keys_to_ignore_on_save = [r"lm_head.weight"]

    def __init__(self, config):
        super().__init__(config)
        self.transformer = GPTNeoModel(config)
        self.jax = config.jax
        self.lm_head = to_gpu(nn.Linear(config.hidden_size, config.vocab_size,
                                        bias=self.jax and not config.rotary_half),
                              config)
        self.config = config
        self.no_emb_tie = config.no_emb_tie
        self.init_weights()

    def setup_pipe(self):
        if self.jax:
            self.transformer.lm_head = self.lm_head
        else:
            self.transformer.lm_head = to_gpu(
                nn.Linear(self.config.hidden_size, self.config.vocab_size,
                          bias=False), self.config)
            self.transformer.lm_head.weight.data = self.lm_head.weight.data.clone()
        self.transformer.setup_pipe()
        self.lm_head = None
        for i in range(len(self.transformer.h)):
            self.transformer.h[i] = None
        self.transformer.lm_head = None
        self.transformer.ln_f = None

    def get_output_embeddings(self):
        if self.jax or self.no_emb_tie:
            return None
        return self.lm_head

    def set_output_embeddings(self, new_embeddings):
        if self.jax or self.no_emb_tie:
            return
        self.lm_head = new_embeddings

    def prepare_inputs_for_generation(self, input_ids, past=None, **kwargs):
        token_type_ids = kwargs.get("token_type_ids", None)
        # only last token for inputs_ids if past is defined in kwargs
        have_past = isinstance(past, torch.Tensor) or past
        if have_past:
            if isinstance(past, torch.Tensor):
                past = past.to(input_ids.device)
            input_ids = input_ids[:, -1].unsqueeze(-1)
            if token_type_ids is not None:
                token_type_ids = token_type_ids[:, -1].unsqueeze(-1)

        attention_mask = kwargs.get("attention_mask", None)
        position_ids = kwargs.get("position_ids", None)

        if attention_mask is not None and position_ids is None:
            # create position_ids on the fly for batch generation
            position_ids = attention_mask.long().cumsum(-1) - 1
            position_ids.masked_fill_(attention_mask == 0, 1)
            if have_past:
                position_ids = position_ids[:, -1].unsqueeze(-1)
        else:
            position_ids = None
        return {
            "input_ids": input_ids,
            "past_key_values": past,
            "use_cache": kwargs.get("use_cache"),
            "position_ids": position_ids,
            "attention_mask": attention_mask,
            "token_type_ids": token_type_ids,
        }

    def prep_for_embs(self, ids, embs):
        if embs is None:
            return ids
        embs = sorted(embs, key=lambda x: x[0])
        offset = 0
        for n, emb in embs:
            emb_len = emb.shape[0]
            if len(emb.shape) == 3:
                emb_len = emb.shape[1]
            filler = torch.full((ids.shape[0], emb_len), 50256).to(
                ids.device).long()
            n += offset
            if n == 0:
                ids = torch.cat((filler, ids), dim=1)
            elif n >= len(ids[0]):
                ids = torch.cat((ids, filler), dim=1)
            else:
                ids = torch.cat((ids[:, 0:n], filler, ids[:, n:]), dim=1)
            offset += emb_len
        return ids

    def remove_embs_padding(self, ids, embs):
        if embs is None:
            return ids
        embs = sorted(embs, key=lambda x: x[0])
        for n, emb in embs:
            emb_len = emb.shape[0]
            if len(emb.shape) == 3:
                emb_len = emb.shape[1]
            if n == 0:
                ids = ids[:, emb_len:]
            elif n >= len(ids[0]):
                ids = ids[:, :-emb_len]
            else:
                ids = torch.cat((ids[:, 0:n], ids[:, n + emb_len:]), dim=1)
        return ids

    @add_start_docstrings_to_model_forward(GPT_NEO_INPUTS_DOCSTRING)
    @add_code_sample_docstrings(
        tokenizer_class=_TOKENIZER_FOR_DOC,
        checkpoint=_CHECKPOINT_FOR_DOC,
        output_type=CausalLMOutputWithCrossAttentions,
        config_class=_CONFIG_FOR_DOC,
    )
    def forward(
            self,
            input_ids=None,
            past_key_values=None,
            attention_mask=None,
            token_type_ids=None,
            position_ids=None,
            head_mask=None,
            inputs_embeds=None,
            labels=None,
            use_cache=None,
            output_attentions=None,
            output_hidden_states=None,
            return_dict=None,
            embs=None,
            hypernetwork=None,
    ):
        r"""
        labels (:obj:`torch.LongTensor` of shape :obj:`(batch_size, sequence_length)`, `optional`):
            Labels for language modeling. Note that the labels **are shifted** inside the model, i.e. you can set
            ``labels = input_ids`` Indices are selected in ``[-100, 0, ..., config.vocab_size]`` All labels set to
            ``-100`` are ignored (masked), the loss is only computed for labels in ``[0, ..., config.vocab_size]``
        """
        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        transformer_outputs = self.transformer(
            input_ids,
            past_key_values=past_key_values,
            attention_mask=attention_mask,
            token_type_ids=token_type_ids,
            position_ids=position_ids,
            head_mask=head_mask,
            inputs_embeds=inputs_embeds,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
            embs=embs,
            hypernetwork=hypernetwork,
        )

        if self.transformer.pipe is None:
            hidden_states = transformer_outputs[0]
            lm_logits = self.lm_head(hidden_states).float()
            out_dtype = hidden_states.dtype
        else:
            lm_logits = transformer_outputs[0].to(input_ids.device)
            out_dtype = lm_logits.dtype
            hidden_states = None

        loss = None
        if labels is not None:
            # Compute loss in fp32 to match with mesh-tf version
            # https://github.com/EleutherAI/gpt-neo/blob/89ce74164da2fb16179106f54e2269b5da8db333/models/gpt2/gpt2.py#L179
            lm_logits = lm_logits.to(torch.float32)
            labels = labels.to(lm_logits.device)

            # Shift so that tokens < n predict n
            shift_logits = lm_logits[..., :-1, :].contiguous()
            shift_labels = labels[..., 1:].contiguous()
            # Flatten the tokens
            loss_fct = CrossEntropyLoss()
            loss = loss_fct(shift_logits.view(-1, shift_logits.size(-1)),
                            shift_labels.view(-1))

            lm_logits = lm_logits.to(out_dtype)
            loss = loss.to(out_dtype)

        if not return_dict:
            output = (lm_logits,) + transformer_outputs[1:]
            return ((loss,) + output) if loss is not None else output

        return CausalLMOutputWithPast(
            loss=loss,
            logits=lm_logits,
            past_key_values=transformer_outputs.past_key_values,
            hidden_states=transformer_outputs.hidden_states,
            attentions=transformer_outputs.attentions,
        )

    @staticmethod
    def _reorder_cache(past: Tuple[Tuple[torch.Tensor]],
                       beam_idx: torch.Tensor) -> Tuple[Tuple[torch.Tensor]]:
        """
        This function is used to re-order the :obj:`past_key_values` cache if
        :meth:`~transformers.PretrainedModel.beam_search` or :meth:`~transformers.PretrainedModel.beam_sample` is
        called. This is required to match :obj:`past_key_values` with the correct beam_idx at every generation step.
        """
        return tuple(
            tuple(past_state.index_select(0, beam_idx.to(past_state.device)) for
                  past_state in layer_past)
            for layer_past in past
        )
