import React, { useState } from 'react'
import { Text } from 'react-native'
import { useRecoilState } from 'recoil'
import { Session } from '../../../shared/globals/state'
import { isNonsenseAllowed } from '../../../shared/util/placebo'
import Checkbox from '../common/checkbox'
import { MainSettingSliderCard } from '../common/editorcard'
import { SettingsSectionHeader, SettingsView } from './settings.style'

export default function AISettings(): JSX.Element {
    const [session, setSession] = useRecoilState(Session)

    const [streamResponses, setStreamResponses] = useState(session.settings.streamResponses)
    const changeStreamResponses = (state: boolean) => {
        setSession({ ...session, settings: { ...session.settings, streamResponses: state } })
        setStreamResponses(state)
    }

    const [delay, setDelay] = useState(session.settings.streamDelay ?? 0)
    const changeDelay = (scale: number) => {
        setSession({ ...session, settings: { ...session.settings, streamDelay: scale } })
        setDelay(scale)
    }

    const [continueToEndOfSentence, setContinueToEndOfSentence] = useState(
        session.settings.continueGenerationToSentenceEnd
    )
    const changeContinueToEndOfSentence = (state: boolean) => {
        setSession({ ...session, settings: { ...session.settings, continueGenerationToSentenceEnd: state } })
        setContinueToEndOfSentence(state)
    }

    const [trimTrailingSpaces, setTrimTrailingSpaces] = useState(session.settings.trimTrailingSpaces)
    const changeTrailingSpaces = (state: boolean) => {
        setSession({ ...session, settings: { ...session.settings, trimTrailingSpaces: state } })
        setTrimTrailingSpaces(state)
    }

    const [prependPreamble, setPrependPreamble] = useState(session.settings.prependPreamble)
    const changePrependPreamble = (state: boolean) => {
        setSession({ ...session, settings: { ...session.settings, prependPreamble: state } })
        setPrependPreamble(state)
    }

    const [force1024Tokens, setForce1024Tokens] = useState(session.settings.force1024Tokens)
    const changeForce1024Tokens = (state: boolean) => {
        setSession({ ...session, settings: { ...session.settings, force1024Tokens: state } })
        setForce1024Tokens(state)
    }

    const [defaultBias, setDefaultBias] = useState(session.settings.defaultBias)
    const changeDefaultBias = (state: boolean) => {
        setSession({ ...session, settings: { ...session.settings, defaultBias: state } })
        setDefaultBias(state)
    }

    const [placebo, setPlacebo] = useState(session.settings.april2022)
    const changePlacebo = (state: boolean) => {
        setSession({ ...session, settings: { ...session.settings, april2022: state } })
        setPlacebo(state)
    }

    return (
        <SettingsView>
            <SettingsSectionHeader>AI Responses</SettingsSectionHeader>
            <Checkbox
                label="Stream AI Responses"
                value={streamResponses ?? true}
                setValue={changeStreamResponses}
                checkedText={`AI responses will be streamed, appearing token by token.`}
                uncheckedText={'Text will appear all at once when generation has finished'}
            />
            {(streamResponses ?? true) && (
                <MainSettingSliderCard
                    title="Streamed Response Delay"
                    hint="Default: 0"
                    onHintClick={() => changeDelay(0)}
                    value={delay}
                    min={0}
                    max={20}
                    step={1}
                    onChange={changeDelay}
                    compact={true}
                />
            )}
            <Checkbox
                label="Continue until End of Sentence"
                value={continueToEndOfSentence ?? true}
                setValue={changeContinueToEndOfSentence}
                checkedText={'Responses will attempt to continue until the end of a sentence is found.'}
                uncheckedText={`Responses will end normally.`}
            />
            <SettingsSectionHeader>Experimental</SettingsSectionHeader>
            <Checkbox
                label="Prepend a preamble in the context"
                value={prependPreamble ?? true}
                setValue={changePrependPreamble}
                checkedText={
                    <Text>
                        The context will have a small preamble prepended.{'\n'}
                        <Text style={{ fontStyle: 'italic' }}>
                            This adds an asterism (⁂), telling the AI a new story is starting. Automatically
                            removed once it reaches a certain length.
                        </Text>
                    </Text>
                }
                uncheckedText={
                    <Text>
                        The context is not prepended with a preamble.{'\n'}
                        <Text style={{ fontStyle: 'italic' }}>
                            This adds an asterism (⁂), telling the AI a new story is starting. Automatically
                            removed once it reaches a certain length.
                        </Text>
                    </Text>
                }
            />
            <Checkbox
                label="Trim excess whitespace"
                value={trimTrailingSpaces ?? true}
                setValue={changeTrailingSpaces}
                checkedText={
                    <Text>
                        Excess whitespace will be trimmed from context before it is sent to the AI.{'\n'}
                        <Text style={{ marginTop: '5px', fontStyle: 'italic' }}>
                            Excess whitespace includes trailing spaces and repeated newline characters.
                        </Text>
                    </Text>
                }
                uncheckedText={
                    <Text>
                        Excess whitespace will not be trimmed from context before it is sent to the AI.\
                        Excess whitespace can significantly degrade output quality.{'\n'}
                        <Text style={{ marginTop: '5px', fontStyle: 'italic' }}>
                            Excess whitespace includes trailing spaces and repeated newline characters.
                        </Text>
                    </Text>
                }
            />
            <Checkbox
                label="Force 1024 token limit"
                value={force1024Tokens ?? false}
                setValue={changeForce1024Tokens}
                checkedText={'The maximum token context limit will be 1024 regardless of subscription tier.'}
                uncheckedText={`The maximum token context limit will be based on your subscription tier (1024 for Tablet, 2048 for Scroll and Opus.)`}
            />
            <Checkbox
                label="Apply default bias"
                value={defaultBias ?? false}
                setValue={changeDefaultBias}
                checkedText={`A default bias will be applied, reducing the likelyhood of dinkus (***) and asterism (⁂) to be generated.`}
                uncheckedText={`No default bias will be applied, not reducing the likelyhood of dinkus (***) and asterism (⁂) to be generated.`}
            />
            {isNonsenseAllowed() && (
                <Checkbox
                    label="Placebo"
                    value={placebo ?? false}
                    setValue={changePlacebo}
                    uncheckedText={`It does nothing.`}
                    checkedText={`It does nothing... right?`}
                />
            )}
        </SettingsView>
    )
}
