﻿import React, { useEffect, useMemo, useState } from 'react'
import { Modal, View } from 'react-native'
import { useSetRecoilState } from 'recoil'
import styled from 'styled-components/native'
import { StoryStatistics, NlpStoryStatistics } from '../../../../shared/data/story/story'
import { GlobalUserContext } from '../../../../shared/globals/globals'
import { StoryUpdate } from '../../../../shared/globals/state'
import {
    BodyLarge400,
    BodyLarge600,
    BodyLarge700M,
    BodyMedium400,
    HeadingSmall600,
    HeadingSmall600M,
} from '../../../styles/fonts'
import { BorderedButton, Button, RegularButtonText } from '../../common/button.style'
import { FullSize } from '../../common/common.style'
import { DotResetIcon, FlattenIcon, ScissorsIcon } from '../../common/icons'
import Spinner from '../../common/spinner'
import { ModalTitleRow } from '../../modals/common'
import { FullModalView } from '../../modals/common.style'
import { WarningModalButton } from '../../modals/deletemodal'
import { StyledStatsIcon } from '../infobar.style'

const Section = styled.View`
    border: solid 1px ${(props) => props.theme.colors.bg3};
    border-radius: 3px;
    padding: 20px 20px 10px;
    margin-vertical: 10px;
`

const DeepSection = styled.View`
    border: solid 1px ${(props) => props.theme.colors.bg3};
    border-radius: 3px;
    padding: 20px 40px;
    margin-bottom: 10px;
`

const SectionName = styled.Text`
    ${HeadingSmall600M}
    color: ${(props) => props.theme.colors.textHeadings};
    margin-top: 10px;
`

const StyledDataRow = styled.View`
    flex-direction: row;
    justify-content: space-between;
    align-items: center;
    padding-bottom: 10px;
`

const Scissors = styled(ScissorsIcon).attrs((props) => ({
    primary: props.theme.colors.warning,
}))`
    top: 2%;
`

const Flatten = styled(FlattenIcon).attrs((props) => ({
    primary: props.theme.colors.warning,
}))`
    top: 2%;
`

const DotReset = styled(DotResetIcon).attrs((props) => ({
    primary: props.theme.colors.warning,
}))`
    top: 2%;
`

function DataRow(props: { name: string; data: number | string; subsection?: boolean }): JSX.Element {
    return props.subsection ? (
        <StyledDataRow>
            <BodyLarge400 style={{ paddingLeft: 20 }}>{props.name}</BodyLarge400>
            <BodyLarge400>{props.data}</BodyLarge400>
        </StyledDataRow>
    ) : (
        <StyledDataRow>
            <BodyLarge600>{props.name}</BodyLarge600>
            <BodyLarge400>{props.data}</BodyLarge400>
        </StyledDataRow>
    )
}

const EntryNumber = styled.Text`
    ${BodyLarge700M}
    width: 24px;
`

const Table = styled.View`
    flex-direction: row;
    justify-content: space-between;
    margin-top: 10px;
`

const Column = styled.View`
    flex-direction: column;
    align-items: flex-start;
`

function TopWords(props: { usedWords: Map<string, number> | undefined }): JSX.Element {
    const [halfLength, setHalfLength] = useState(0)

    const slices: [string, number][][] | undefined = useMemo(() => {
        if (!props.usedWords) return undefined
        const allWords = [...props.usedWords.entries()].sort((a, b) => b[1] - a[1]).slice(0, 10)
        const newHalfLength = Math.floor(allWords.length / 2)
        setHalfLength(newHalfLength)
        return [allWords.slice(0, newHalfLength), allWords.slice(newHalfLength)]
    }, [props.usedWords])

    return (
        <DeepSection>
            <HeadingSmall600>Most Used Words</HeadingSmall600>
            {slices ? (
                <Table>
                    {slices.map((section, index) => (
                        <Column key={index}>
                            {section.map((entry, id) => {
                                return (
                                    <StyledDataRow key={id}>
                                        <EntryNumber>{index * halfLength + id}.</EntryNumber>
                                        <BodyLarge400>{entry[0]}</BodyLarge400>
                                    </StyledDataRow>
                                )
                            })}
                        </Column>
                    ))}
                </Table>
            ) : (
                <BodyLarge400>???</BodyLarge400>
            )}
        </DeepSection>
    )
}

export default function StoryStats(props: { selectedStory: string }): JSX.Element {
    // TODO: stats for document
    const setStoryUpdate = useSetRecoilState(StoryUpdate(''))
    const currentStory = GlobalUserContext.stories.get(props.selectedStory)
    const currentStoryContent = GlobalUserContext.storyContentCache.get(props.selectedStory)
    const [statsModalVisible, setStatsModalVisible] = useState(false)
    const [stats, setStats] = useState(new StoryStatistics())
    const [nlpStats, setNlpStats] = useState<NlpStoryStatistics | undefined>()

    const [statsLoaded, setStatsLoaded] = useState(false)

    const [trimWarningModal, setTrimWarningModal] = useState(false)
    const [flattenWarningModal, setFlattenWarningModal] = useState(false)
    const [resetToPromptWarningModal, setResetToPromptWarningModal] = useState(false)

    useEffect(() => {
        const generateStats = async () => {
            if (!currentStoryContent) {
                return
            }
            currentStoryContent.story?.calculateStoryStatistics().then((result) => {
                setStats(result)
                setStatsLoaded(true)
            })
        }

        if (statsModalVisible) {
            generateStats()
        }
    }, [currentStoryContent, currentStoryContent?.story, statsModalVisible])

    if (!currentStory || !currentStoryContent || !currentStoryContent.story) {
        return <></>
    }

    return (
        <BorderedButton
            onPress={() => setStatsModalVisible(true)}
            color={'transparent'}
            style={{ marginBottom: 10 }}
        >
            <StyledStatsIcon />
            <BodyLarge600>View Story Stats</BodyLarge600>
            <Modal visible={statsModalVisible} onRequestClose={() => setStatsModalVisible(false)}>
                <ModalTitleRow title={'Stats'} onClose={() => setStatsModalVisible(false)} />
                <FullModalView>
                    {!statsLoaded ? (
                        <FullSize style={{ justifyContent: 'center' }}>
                            <Spinner visible={true} style={{ alignSelf: 'center' }} />
                        </FullSize>
                    ) : (
                        <View style={{ paddingVertical: 20 }}>
                            <SectionName>Data</SectionName>
                            <Section>
                                <DataRow name={'Data Blocks'} data={stats.dataBlocks} />
                                <DataRow name={'User Input'} data={stats.userBlocks} />
                                <DataRow name={'Edits'} data={stats.editBlocks} />
                                <DataRow
                                    name={'AI Responses'}
                                    data={stats.responseBlocks}
                                    subsection={true}
                                />
                                <DataRow
                                    name={'Lorebook Entries'}
                                    data={currentStoryContent.lorebook.entries.length}
                                    subsection={true}
                                />
                                <DataRow
                                    name={'Ephemeral Entries'}
                                    data={currentStoryContent.ephemeralContext.length}
                                    subsection={true}
                                />
                            </Section>
                            <SectionName>Structure</SectionName>
                            <Section>
                                <DataRow name={'Current Step'} data={stats.currentStep} />
                                <DataRow name={'Furthest Step'} data={stats.furthestStep} />
                                <DataRow name={'Dead Ends'} data={stats.deadEnds} />
                                <DataRow name={'No Retry Streak'} data={stats.noRetryStreak} />
                                <DataRow
                                    name={'Longest Abandoned Branch'}
                                    data={stats.longestAbandonedBranch}
                                />
                                <DataRow name={'Most Retries'} data={stats.mostRetries} />
                            </Section>
                            <SectionName>Writing</SectionName>
                            <Section>
                                <DataRow name={'Characters'} data={stats.characters} />
                                <DataRow
                                    name={'AI Responses'}
                                    data={stats.responseBlocks}
                                    subsection={true}
                                />
                                <DataRow
                                    name={'Lorebook Entries'}
                                    data={currentStoryContent.lorebook.entries.length}
                                    subsection={true}
                                />
                                <DataRow
                                    name={'Ephemeral Entries'}
                                    data={currentStoryContent.ephemeralContext.length}
                                    subsection={true}
                                />
                                <DataRow name={'Paragraphs'} data={stats.paragraphs} />
                            </Section>
                            <SectionName>Additional Stats</SectionName>
                            <Button
                                style={{ marginTop: 10 }}
                                onPress={async () => {
                                    currentStoryContent.story!.calculateNlpStats().then((result) => {
                                        setNlpStats(result)
                                    })
                                }}
                            >
                                <RegularButtonText>Generate Additional Stats</RegularButtonText>
                            </Button>
                            <Section>
                                <DataRow name={'Words'} data={nlpStats?.words ?? '?'} />
                                <DataRow name={'Sentences'} data={nlpStats?.sentences ?? '?'} />
                            </Section>
                            <TopWords usedWords={nlpStats?.usedWords} />
                            <SectionName>Experimental</SectionName>
                            <BodyMedium400>
                                WARNING: The following options are experimental and could result in permanent
                                corruption of your story. Creating a backup first is heavily advised.
                            </BodyMedium400>
                            <WarningModalButton
                                visible={trimWarningModal}
                                setVisible={setTrimWarningModal}
                                buttonText={'Trim Story'}
                                label={'Trim your Story?'}
                                description={`Are you sure you want to trim the history of "${currentStory.title}" to its prompt? This will delete all story branches. This cannot be reversed.`}
                                confirmText={'Trim it!'}
                                icon={<Scissors />}
                                onConfirm={() => {
                                    currentStoryContent.story!.trimBranches()
                                    /** eventBus.trigger(
                                        createEditorEvent(
                                            new EditorLoadEvent(
                                                currentStoryContent,
                                                currentStory
                                            )
                                        )
                                    ) */
                                    setStoryUpdate(currentStory.save())
                                    setTrimWarningModal(false)
                                }}
                            />
                            <WarningModalButton
                                visible={flattenWarningModal}
                                setVisible={setFlattenWarningModal}
                                buttonText={'Flatten Story'}
                                label={'Flatten your Story?'}
                                description={`Are you sure you want to flatten "${currentStory.title}" to its prompt? This will delete its entire history. This cannot be reversed.`}
                                confirmText={'Flatten it!'}
                                icon={<Flatten />}
                                onConfirm={() => {
                                    currentStoryContent.story!.flattenStory()
                                    /** eventBus.trigger(
                                        createEditorEvent(
                                            new EditorLoadEvent(
                                                currentStoryContent,
                                                currentStory
                                            )
                                        )
                                    ) */
                                    setStoryUpdate(currentStory.save())
                                    setTrimWarningModal(false)
                                }}
                            />
                            <WarningModalButton
                                visible={resetToPromptWarningModal}
                                setVisible={setResetToPromptWarningModal}
                                buttonText={'Reset to Prompt'}
                                label={'Reset your Story to Prompt?'}
                                description={`Are you sure you want to reset "${currentStory.title}" to its prompt? This will delete every input/output up til now. This cannot be reversed.`}
                                confirmText={'Clear it!'}
                                icon={<DotReset />}
                                onConfirm={() => {
                                    currentStoryContent.story!.resetToPrompt()
                                    currentStoryContent.didGenerate = false
                                    currentStory.textPreview = currentStoryContent
                                        .story!.getText()
                                        .slice(0, 250)
                                    /** eventBus.trigger(
                                        createEditorEvent(
                                            new EditorLoadEvent(
                                                currentStoryContent,
                                                currentStory
                                            )
                                        )
                                    ) */
                                    setStoryUpdate(currentStory.save())
                                    setResetToPromptWarningModal(false)
                                }}
                            />
                        </View>
                    )}
                </FullModalView>
            </Modal>
        </BorderedButton>
    )
}
