﻿import React, { useState } from 'react'
import { useRecoilValue, useSetRecoilState } from 'recoil'
import { serialize } from 'serializr'
import styled from 'styled-components/native'
import { storyAsScenario } from '../../../../shared/data/story/scenario'
import { StoryContainer } from '../../../../shared/data/story/storycontainer'
import { AIModule } from '../../../../shared/data/story/storysettings'
import { GlobalUserContext } from '../../../../shared/globals/globals'
import { CustomModules, ScreenshotModalState, Session } from '../../../../shared/globals/state'
import { deserialize } from '../../../../shared/util/serialization'
import { downloadTextFile } from '../../../util/browser'
import { toast } from '../../../util/toast'
import Clipboard from '@react-native-clipboard/clipboard'
import { View } from 'react-native'
import {
    LightColorButton,
    RegularButtonText,
    SubmitButton,
    SubmitButtonText,
} from '../../common/button.style'
import { ClipBoardIcon, ExportIcon } from '../../common/icons'
import { BodyLarge700 } from '../../../styles/fonts'
import { FlexRow } from '../../common/common.style'
import { getUserSetting } from '../../../../shared/data/user/settings'
import useDownloadStoryJson from '../../../../shared/hooks/useDownloadStory'

export function copyToClipboard(text: string): void {
    Clipboard.setString(text)
}

export default function StoryExporter(props: { selectedStory: string }): JSX.Element {
    const session = useRecoilValue(Session)
    const customModules = useRecoilValue(CustomModules)

    const currentStoryMetadata = GlobalUserContext.stories.get(props.selectedStory)
    const currentStoryContent = GlobalUserContext.storyContentCache.get(props.selectedStory)
    const downloadStoryJson = useDownloadStoryJson(props.selectedStory)

    const downloadScenarioJson = () => {
        const scenario = storyAsScenario(props.selectedStory, customModules)
        if (!scenario || !currentStoryMetadata) {
            return
        }
        scenario.author = getUserSetting(session.settings, 'penName') ?? ''
        downloadTextFile(
            scenario.serialize(true),
            `${currentStoryMetadata.title.slice(0, 40)} (${new Date().toISOString()}).scenario`
        )
    }

    const downloadStoryPlainText = () => {
        if (!currentStoryContent || !currentStoryMetadata) {
            return
        }
        downloadTextFile(
            currentStoryContent.getStoryText(),
            `${currentStoryMetadata.title.slice(0, 40)} (${new Date().toISOString()}).txt`
        )
    }

    const copyStoryToClipboard = async () => {
        if (!currentStoryContent || !currentStoryMetadata) {
            return
        }

        let container = StoryContainer.bundle(currentStoryMetadata, currentStoryContent)
        const serialized = container.serialize()
        container = StoryContainer.deserialize(serialized)
        container.metadata.remoteId = undefined
        container.metadata.remoteStoryId = undefined

        if (currentStoryContent.settings.prefix && currentStoryContent.settings.prefix.includes(':')) {
            let aiModule = customModules.find((e: AIModule) => e.id === currentStoryContent.settings.prefix)
            if (aiModule) {
                aiModule = deserialize(AIModule, serialize(AIModule, aiModule))
                aiModule.remoteId = ''
                container.content.settings.aiModule = aiModule
                container.content.settings.aiModule.remoteId = ''
            }
        }

        copyToClipboard(container.serialize())
        toast('Story copied to clipboard')
    }

    const setScreenshotState = useSetRecoilState(ScreenshotModalState)
    const openScreenshotTool = async () => {
        if (!currentStoryContent || !currentStoryMetadata) {
            return
        }

        setScreenshotState({
            open: true,
            start: 0,
            end: currentStoryContent.getStoryText().length,
        })
    }

    const [menuOpen, setMenuOpen] = useState(false)

    return (
        <View style={{ marginBottom: 30 }}>
            <BodyLarge700 style={{ marginBottom: 10 }}>Export Story</BodyLarge700>
            <FlexRow>
                <SubmitButton
                    style={{ marginTop: 0, marginRight: 12, width: '50%', flexShrink: 1 }}
                    onPress={downloadStoryJson}
                >
                    <>
                        <ToFile />
                        <SubmitButtonText>To File</SubmitButtonText>
                    </>
                </SubmitButton>
                <LightColorButton style={{ width: '50%', flexShrink: 1 }} onPress={downloadScenarioJson}>
                    <>
                        <AsScenario />
                        <RegularButtonText>As Scenario</RegularButtonText>
                    </>
                </LightColorButton>
            </FlexRow>
            <LightColorButton style={{ marginTop: 10 }} onPress={copyStoryToClipboard}>
                <>
                    <ToClipBoard />
                    <RegularButtonText>To Clipboard</RegularButtonText>
                </>
            </LightColorButton>
        </View>
    )
}

export const ToFile = styled(ExportIcon).attrs((props) => ({
    primary: props.theme.colors.bg0,
}))`
    margin-right: 8px;
    padding: 8px;
`

export const AsScenario = styled(ExportIcon).attrs((props) => ({
    primary: props.theme.colors.textMain,
}))`
    margin-right: 8px;
    padding: 8px;
`

export const ToClipBoard = styled(ClipBoardIcon).attrs((props) => ({
    primary: props.theme.colors.textMain,
}))`
    margin-right: 8px;
    padding: 8px;
`
