import { MutableRefObject, RefObject } from 'react'
import { TextInputTextInputEventData } from 'react-native'
import { DataOrigin } from '../../../shared/components/editor/glue'
import { Section, SectionId, SectionType } from '../../../shared/data/document/section'
import { uniqueid, UniqueId } from '../../../shared/data/document/util'
import { Paragraph, Segment } from './model'
import { Change, Document } from '../../../shared/data/document/document'
import { ParagraphHandle, paragraphToSection, sectionsToParagraphs, sectionToParagraph } from './convert'
import { getOrigin, glueSegments, ignoreDot } from './util'
import { defaultSections, defaultSegments } from './testdata'
import { EditorViewHandle } from './editorview'
import { ChangeMap, HistoryStepType } from '../../../shared/data/document/history'
import { serialize } from 'serializr'

export class EditorModel {
    private editorRef: RefObject<EditorViewHandle>
    private documentRef: MutableRefObject<Document>
    private baseMarkRef: MutableRefObject<DataOrigin>
    paragraphs: Map<SectionId, Paragraph>
    order: Array<SectionId>
    private usedSegIDs: UniqueId[]

    constructor(
        editorRef: RefObject<EditorViewHandle>,
        documentRef: MutableRefObject<Document>,
        baseMarkRef: MutableRefObject<DataOrigin>
    ) {
        this.editorRef = editorRef
        this.documentRef = documentRef
        this.baseMarkRef = baseMarkRef
        this.paragraphs = new Map()
        this.order = []
        this.usedSegIDs = []
        this.loadDocument()
    }

    /**
     * Document Handling
     */
    loadDocument() {
        if (!this.documentRef.current) return
        this.paragraphs.clear()
        this.usedSegIDs = []
        this.order = sectionsToParagraphs(this.documentRef.current).map(({ id, paragraph }) => {
            paragraph.segments.forEach((seg) => seg.assignID(this.usedSegIDs))
            this.paragraphs.set(id, paragraph)
            return id
        })
        this.editorRef.current?.redraw()
    }

    stageChanges(...affectedPars: UniqueId[]) {
        if (!this.editorRef.current) return
        const changes = new Map<SectionId, Change<Section>>()
        affectedPars.forEach((id) => {
            const paragraph = this.paragraphs.get(id)
            const index = this.order.indexOf(id)
            const after = index > 0 ? this.order[index - 1] : undefined
            changes.set(id, {
                changedSection: paragraph ? paragraphToSection(paragraph) : undefined,
                after: after ?? 0,
            })
        })
        this.editorRef.current.onModelChanged(changes)
    }

    /**
     * Model Manipulation
     */
    private insertSection(id: SectionId, content: Section, after?: SectionId): number {
        const afterPar = after ? this.paragraphs.get(after) : undefined
        const newPar = sectionToParagraph(content, afterPar ? afterPar.start + afterPar.length : 0)
        const index = after ? this.order.indexOf(after) + 1 : 0
        newPar.segments.forEach((seg) => seg.assignID(this.usedSegIDs))
        if (!this.order.includes(id)) this.order.splice(index, 0, id)
        this.paragraphs.set(id, newPar)
        return index
    }

    private removeSection(id: SectionId, after?: SectionId) {
        const oldSegIDs = this.paragraphs.get(id)?.segments.map((seg) => seg.id) ?? []
        this.usedSegIDs = this.usedSegIDs.filter((id) => !oldSegIDs.includes(id))
        this.order.splice(after ? this.order.indexOf(after) + 1 : 0, 1)
        this.paragraphs.delete(id)
        return after ? this.order.indexOf(after) : -1
    }

    private updateSection(id: SectionId, content: Section): number {
        const oldPar = this.paragraphs.get(id)
        const oldSegIDs = oldPar?.segments.map((seg) => seg.id) ?? []
        const newPar = sectionToParagraph(content, oldPar?.start ?? 0)
        newPar.segments.forEach((seg) => seg.assignID(this.usedSegIDs))
        this.usedSegIDs = this.usedSegIDs.filter((id) => !oldSegIDs.includes(id))
        this.paragraphs.set(id, newPar)
        return this.order.indexOf(id)
    }

    check() {
        const actualPars = sectionsToParagraphs(this.documentRef.current)
        const handles: ParagraphHandle[] = this.order.reduce((handles: ParagraphHandle[], id) => {
            const par = this.paragraphs.get(id)
            if (!par) return handles
            handles.push({
                id: id,
                paragraph: new Paragraph(
                    par.segments.map((seg) => new Segment(seg.origin, seg.data, seg.style, seg.start)),
                    par.start,
                    par.length,
                    par.dimensions
                ),
            })
            return handles
        }, [])
        //console.log(JSON.stringify(actualPars))
        //console.log(JSON.stringify(handles))
        handles.forEach((value, index) => {
            const correspondingIndex = actualPars.findIndex((par) => par.id === value.id)
            if (correspondingIndex < 0) {
                console.warn(`Paragraph with id ${value.id} should not exist.`)
                return
            }
            //Note: probably doesn't compare styles accurately.
            let mismatch = false
            value.paragraph.segments.forEach((seg, segIndex) => {
                const actualSeg = actualPars[correspondingIndex].paragraph.segments[segIndex]
                if (!actualSeg) {
                    console.warn(
                        `Extra segment in paragraph with id ${value.id}. Value: ${JSON.stringify(seg)}.`
                    )
                    mismatch = true
                } else if (JSON.stringify(seg) !== JSON.stringify(actualSeg)) {
                    console.warn(
                        `Segment mismatch in paragraph with id ${
                            value.id
                        }. Segment with value ${JSON.stringify(seg)} should be of value ${JSON.stringify(
                            actualSeg
                        )}.`
                    )
                    mismatch = true
                }
            })
            const currentSection = this.documentRef.current.getSection(value.id)
            if (mismatch && currentSection?.type === SectionType.text) {
                console.warn(
                    `Paragraph start: ${value.paragraph.start}, length: ${value.paragraph.length}.`,
                    'Section content before conversion:',
                    currentSection.text,
                    [...(currentSection.meta.get(1)?.entries() ?? [])].reduce(
                        (metas, [id, data]) =>
                            metas +
                            `Meta: ${id}, data: ${data.data}, length: ${data.length}, position: ${data.position}. `,
                        ''
                    )
                )
                sectionToParagraph(currentSection, 0)
            }
            if (index !== correspondingIndex)
                console.warn(`Paragraph with id ${value.id} is not ordered correctly.`)
        })
        actualPars.forEach((value) => {
            if (!handles.find((handle) => value.id === handle.id))
                console.warn(`Paragraph with id ${value.id} is missing.`)
        })
        console.log('Checked.')
    }

    applyChanges(changes: ChangeMap, undo?: boolean) {
        //console.log('Before:', JSON.stringify(this.paragraphs.entries()))

        const before: ParagraphHandle[] = [...this.paragraphs.entries()].map(([id, par]) => ({
            id: id,
            paragraph: par,
        }))
        //console.log('Before:', before)
        //console.log('Changes:', changes)
        const oldOrder = [...this.order]
        let parsToRefresh: SectionId[] = []
        changes.forEach((change, id) => {
            switch (change.type) {
                case HistoryStepType.create:
                    undo
                        ? this.removeSection(id, change.after)
                        : this.insertSection(id, change.section, change.after)
                    break
                case HistoryStepType.remove:
                    undo
                        ? this.insertSection(id, change.previous, change.after)
                        : this.removeSection(id, change.after)
                    break
                case HistoryStepType.update:
                    const section = this.documentRef.current.getSection(id)
                    if (section) this.updateSection(id, section)
                    else console.warn('Could not find section to update.')
                    parsToRefresh.push(id)
                    break
            }
        })
        this.startsOnward()
        //console.log('After:', this.paragraphs, this.order)
        this.editorRef.current?.paragraphRefresh(...parsToRefresh)
        JSON.stringify(oldOrder) !== JSON.stringify(this.order) && this.editorRef.current?.redraw()
        if (this.editorRef.current) this.editorRef.current.empty = this.order.length <= 0
        //this.check()
    }

    private addParagraph(paragraph: Paragraph) {
        let id = uniqueid()
        while (this.paragraphs.has(id)) id = uniqueid()
        this.paragraphs.set(id, paragraph)
        return id
    }

    private startsOnward(id?: UniqueId, startSegIndex?: number) {
        if (id && (!this.order.includes(id) || !this.paragraphs.has(id))) {
            console.warn('Could not calculate starts. The first paragraph could not be found.')
            return
        }

        let counter = 0

        const startPar = id ? this.paragraphs.get(id) : undefined
        if (startPar) {
            startPar.updateSegStarts(startSegIndex && startSegIndex > 0 ? startSegIndex - 1 : undefined)
            counter = startPar.start + startPar.length
        }

        const startIndex = id ? this.order.indexOf(id) + 1 : 0
        for (let i = startIndex; i < this.order.length; i++) {
            const par = this.paragraphs.get(this.order[i])
            if (!par) {
                console.warn('Tried to update start of invalid paragraph.')
                continue
            }
            par.start = counter
            par.updateSegStarts()
            counter += par.length
        }
    }

    /**
     * Checks
     */
    private containersFromPosition(position: number): [UniqueId | undefined, Segment | undefined] {
        //get paragraph
        const paragraphID = this.order.find((id) => {
            const paragraph = this.paragraphs.get(id)
            return paragraph && paragraph.start <= position && position < paragraph.start + paragraph.length
        })
        if (!paragraphID) {
            console.warn('Could not find paragraph')
            return [undefined, undefined]
        }
        //get segment
        const par = this.paragraphs.get(paragraphID)
        const segment =
            par && par.segments.length > 0
                ? par.segments.find(
                      (seg) => seg.start < position && position <= seg.start + seg.data.length
                  ) ?? par.segments[0]
                : undefined
        return [paragraphID, segment]
    }

    /**
     * Input Handling
     */
    processInput(rawData: TextInputTextInputEventData, fromAI?: boolean) {
        if (!this.editorRef.current) return
        //console.log('Event Data:', rawData)
        const data = !this.editorRef.current.empty
            ? ignoreDot(rawData, this.editorRef.current.restoreFirstChar)
            : rawData

        //If the dot is removed and nothing else, don't even bother handling it.
        if (data.text.length === 0 && data.range.end - data.range.start === 0) return

        let [startParID, startSeg] = this.containersFromPosition(data.range.start)
        let oldOder = [...this.order]
        let segsToRefresh: Segment[] = []
        let parsToRefresh: UniqueId[] = []
        let affectedSections: UniqueId[] = []
        let existingSegment = true
        let deletedSegsIDs: UniqueId[] = []

        let baseMark = fromAI ? DataOrigin.ai : this.baseMarkRef.current

        //console.log('Processed Data:', data)

        //for detecting paragraph breaks
        const parts = data.text.split('\n')

        //If no paragraph can be found (position out of bounds), the last paragraph is selected.
        //If there are no paragraphs, then the first paragraph is created.
        if (!startParID) {
            startParID =
                this.order.length > 0 ? this.order[this.order.length - 1] : this.addParagraph(new Paragraph())
            if (this.order.length === 0) this.order.push(startParID)
        }

        const startPar = this.paragraphs.get(startParID) //could get from containers function instead
        const startParIndex = this.order.indexOf(startParID)
        if (!startPar) {
            console.warn('Start paragraph could not be found')
            return
        }

        if (!startSeg) {
            //Assign startSeg to first or new Segment, then carry on with regular functions.
            startSeg = startPar.segments.length > 0 ? startPar.segments[0] : new Segment(baseMark)
            if (startPar.segments.length === 0) {
                //In this case, a new segment was created and must be handled.
                startSeg.assignID(this.usedSegIDs)
                startPar.segments.push(startSeg)
                parsToRefresh.push(startParID)
                existingSegment = false
            }
        }

        const startSegIndex = startPar.segments.indexOf(startSeg)
        const relStart = data.range.start - startSeg.start
        let [endParID, endSeg] = this.containersFromPosition(data.range.end)
        if (!endParID) endParID = startParID
        if (!endSeg) endSeg = startSeg
        const relEnd = data.range.end - endSeg.start

        if (
            existingSegment &&
            startParID === endParID &&
            startSeg.id === endSeg?.id &&
            (startSeg.origin === baseMark || (startSeg.origin === DataOrigin.edit && !fromAI)) &&
            parts.length < 2
        ) {
            //Segment Edit
            startSeg.data =
                startSeg.data.substring(0, relStart) +
                data.text +
                startSeg.data.substring(data.range.end - startSeg.start)
            if (startSeg.data.trim().length > 0) {
                //simply modified
                segsToRefresh.push(startSeg)
                affectedSections.push(startParID) //since it won't be added to parsToRefresh
                //console.log('Modifying segment:', startSeg)
            } else {
                //deleted
                const index = startPar.segments.indexOf(startSeg)
                const withNeighbors = index > 0 ? startPar.segments.slice(index - 1, index + 2) : [startSeg]
                const newSegs = glueSegments(withNeighbors)
                newSegs.forEach((seg) => seg.assignID(this.usedSegIDs))

                deletedSegsIDs.push(
                    ...startPar.segments
                        .splice(
                            withNeighbors.length > 1 ? index - 1 : index,
                            withNeighbors.length,
                            ...newSegs
                        )
                        .reduce((ids: UniqueId[], seg) => {
                            if (!newSegs.includes(seg)) ids.push(seg.id)
                            return ids
                        }, [])
                )
                parsToRefresh.push(startParID)
                console.log('Deleting segment')
            }
            startPar?.updateLength()
        } else {
            //Complex Edit
            const endPar = this.paragraphs.get(endParID) ?? startPar
            const endSegIndex = endPar.segments.indexOf(endSeg)
            const endParIndex = this.order.indexOf(endParID)

            const middlePars =
                endParIndex - startParIndex > 1 ? this.order.slice(startParIndex + 1, endParIndex) : []

            const startParRemSegs = startPar.segments.slice(startSegIndex)

            //Origin. Prev and Next are considered to be user if outside of the paragraph.
            const origin =
                !fromAI && relEnd < endSeg.data.length
                    ? getOrigin(
                          startPar === endPar
                              ? startPar.segments.slice(startSegIndex, endSegIndex + 1)
                              : [
                                    ...startParRemSegs,
                                    ...middlePars.flatMap((parID) => {
                                        const par = this.paragraphs.get(parID)
                                        return par ? par.segments : []
                                    }),
                                    ...endPar.segments.slice(0, endSegIndex + 1),
                                ],
                          baseMark
                      )
                    : baseMark
            console.log(data.range.end < endSeg.start + startSeg.data.length)

            //New Segments to be inserted, can be built from existing segments
            const newSegs = glueSegments(
                parts.length > 1
                    ? [
                          new Segment(
                              startSeg.origin,
                              startSeg.data.substring(0, relStart),
                              startSeg.style,
                              startSeg.start
                          ),
                          new Segment(origin, parts.shift()),
                      ]
                    : [
                          new Segment(
                              startSeg.origin,
                              startSeg.data.substring(0, relStart),
                              startSeg.style,
                              startSeg.start
                          ),
                          new Segment(origin, parts.shift()),
                          new Segment(endSeg.origin, endSeg.data.substring(relEnd), endSeg.style),
                          ...endPar.segments.slice(endSegIndex + 1),
                      ]
            ) //empty segments are thrown out

            const lastPart = parts.pop()
            const extraPars: Paragraph[] =
                typeof lastPart === 'string'
                    ? [
                          ...parts.map((part) => new Paragraph([new Segment(origin, part)])),
                          new Paragraph(
                              [
                                  ...glueSegments([
                                      new Segment(origin, lastPart),
                                      new Segment(endSeg.origin, endSeg.data.substring(relEnd), endSeg.style),
                                  ]),
                                  ...endPar.segments.slice(endSegIndex + 1),
                              ].filter((seg) => seg.data.length > 0)
                          ),
                      ]
                    : []

            //ID assignment. Old IDs will not be reused.
            newSegs.forEach((seg) => seg.assignID(this.usedSegIDs))
            const newParIDs: UniqueId[] = extraPars.map((par) => {
                par.segments.forEach((seg) => seg.assignID(this.usedSegIDs))
                return this.addParagraph(par)
            })

            //need to implement glue step and skip step
            deletedSegsIDs.push(
                ...startPar.segments
                    .splice(startSegIndex, startParRemSegs.length, ...newSegs)
                    .reduce((ids: UniqueId[], seg) => {
                        if (!newSegs.includes(seg)) ids.push(seg.id)
                        return ids
                    }, [])
            )
            startPar.updateLength()
            this.order
                .splice(
                    startParIndex,
                    startParID === endParID ? 1 : [startParID, ...middlePars, endParID].length,
                    startParID,
                    ...newParIDs
                )
                .slice(1)
                .forEach((id) => {
                    deletedSegsIDs.push(...(this.paragraphs.get(id)?.segments.map((seg) => seg.id) ?? []))
                    affectedSections.push(id)
                    this.paragraphs.delete(id)
                })
            parsToRefresh.push(startParID)
            affectedSections.push(...newParIDs) //parsToRefresh gets combined to affectedSections down the line, no problem
        }

        const newCursorPosition = rawData.range.start + rawData.text.length
        this.editorRef.current.selection = { start: newCursorPosition, end: newCursorPosition }

        //Apply changes
        this.startsOnward(startParID, startSegIndex > 0 ? startSegIndex - 1 : undefined)
        this.editorRef.current.segmentRefresh(...segsToRefresh)
        this.editorRef.current.paragraphRefresh(...parsToRefresh)
        this.usedSegIDs.filter((id) => !deletedSegsIDs.includes(id))
        if (JSON.stringify(this.order) !== JSON.stringify(oldOder)) this.editorRef.current.redraw()
        this.stageChanges(
            ...parsToRefresh.flatMap((id) => (affectedSections.includes(id) ? [] : [id])),
            ...affectedSections
        )
    }

    /**
     * Component Interaction
     */

    getVisuals(id: SectionId) {
        return this.paragraphs.get(id)?.segments.map((seg) => seg.getVisual())
    }

    /**
     * For Testing Purposes.
     */
    initTestData = async () => {
        this.order = []
        this.paragraphs.clear()
        let counter = 0
        for (const section of defaultSections) {
            let paragraph = new Paragraph()
            paragraph.start = counter
            section.segments.forEach((value) => {
                const newSegment = new Segment(
                    defaultSegments[value].origin,
                    defaultSegments[value].data,
                    defaultSegments[value].style
                )
                newSegment.assignID(this.usedSegIDs)
                paragraph.segments.push(newSegment)
            })
            paragraph.segments = glueSegments(paragraph.segments)
            counter += paragraph.updateLength()
            paragraph.updateSegStarts()
            this.order.push(this.addParagraph(paragraph))
        }
    }

    defaultData() {
        defaultSections.forEach((section, sectionIndex) => {
            const paragraph = this.paragraphs.get(this.order[sectionIndex] ?? -1)
            section.segments.forEach((defaultSegmentIndex, segmentIndex) => {
                const defaultData = defaultSegments[defaultSegmentIndex]
                const segment = paragraph?.segments[segmentIndex]
                if (segment && defaultData) {
                    segment.data = defaultData.data
                    segment.origin = defaultData.origin
                    segment.style = defaultData.style
                }
            })
        })
    }

    checkValues() {
        if (this.order.length === 0) return
        const oldParData = this.order.map((id) => this.paragraphs.get(id))
        const firstPar = this.paragraphs.get(this.order[0])
        if (firstPar) firstPar.start = 0
        this.order.forEach((id) => this.paragraphs.get(id)?.updateLength())
        this.startsOnward(this.order[0])
        console.log(
            serialize(oldParData) === serialize(this.order.map((id) => this.paragraphs.get(id)))
                ? 'matches'
                : 'does not match'
        )
    }

    refreshSegments() {
        this.editorRef.current?.segmentRefresh(
            ...this.order.flatMap((paragraphID) => this.paragraphs.get(paragraphID)?.segments ?? [])
        )
    }

    logs() {
        console.log(this.order)
        this.order.forEach((id) => console.log(this.paragraphs.get(id)))
    }
}
