import React, { MutableRefObject, useCallback, useEffect, useRef } from 'react'
import { useRecoilCallback } from 'recoil'
import styled from 'styled-components/native'
import { Action } from 'typescript-fsa'
import { createEditorEvent, EditorEvent, EditorEventType } from '../../../shared/components/editor/events'
import { DataOrigin } from '../../../shared/components/editor/glue'
import { trimBrokenUnicode } from '../../../shared/data/ai/processresponse'
import { Document } from '../../../shared/data/document/document'
import { EventHandler } from '../../../shared/data/event/eventhandling'
import { LogProbs } from '../../../shared/data/request/remoterequest'
import { StoryContent, StoryMetadata } from '../../../shared/data/story/storycontainer'
import { getUserSetting, UserSettings } from '../../../shared/data/user/settings'
import { eventBus } from '../../../shared/globals/events'
import {
    InputModes,
    IPLimitModal,
    SelectedInputMode,
    SessionValue,
    StoryUpdate,
    SubscriptionDialogOpen,
    TrialUsedModal,
} from '../../../shared/globals/state'
import {
    GenerateError,
    GenerateErrorType,
    RequestWrapper,
    useGenerate,
} from '../../../shared/hooks/useGenerate'
import { useLogout } from '../../hooks/useLogout'
import { BodyLarge400 } from '../../styles/fonts'
import { logDebug, logWarning } from '../../util/browser'
import { toast } from '../../util/toast'
import { EditorControls, EditorControlsHandle } from './controls'
import { Editor, EditorHandle, EditorId } from './editor'

interface EditorContainerProps {
    story: StoryContent
    meta: StoryMetadata
}
export default function EditorContainer({ story, meta }: EditorContainerProps): JSX.Element {
    const editorRef = useRef<EditorHandle>(null)
    const controlsRef = useRef<EditorControlsHandle>(null)
    const blockedRef = useRef(false)
    const didGenerateRef = useRef(0)

    const relayDocumentChange = useRecoilCallback(
        ({ set }) =>
            () => {
                meta.textPreview = story.getStoryText().slice(0, 250)
                set(StoryUpdate(meta.id), meta.save())
            },
        [meta, story]
    )

    const updateState = useCallback(
        (document: Document) => {
            if (controlsRef.current)
                controlsRef.current.state = {
                    blocked: blockedRef.current,
                    canUndo: document.canPopHistory(),
                    canRedo: document.canDescendHistory(),
                    canRetry: document.canPopHistory() && didGenerateRef.current > 0,
                    branches: [...document.getDescendents()].map((id) => ({
                        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                        node: document.getDescendent(id)!,
                        preferred: document.getHistoryNode().route === id,
                    })),
                }
            if (editorRef.current)
                editorRef.current.state = {
                    blocked: blockedRef.current,
                    baseMark: story.didGenerate ? DataOrigin.user : DataOrigin.prompt,
                }
        },
        [story.didGenerate]
    )

    useEffect(() => {
        if (story.document) {
            updateState(story.document)
        }
    }, [story.document, /** updateLinks */ updateState])

    const handleEvent = useCallback(
        (event: EditorEvent) => {
            if (!story.document) return
            switch (event.type) {
                case EditorEventType.load: {
                    didGenerateRef.current = 0
                    break
                }
                case EditorEventType.decorate: {
                    //updateLinks(story.document)
                    console.log('decorate event')
                    break
                }
            }
        },
        [story /** updateLinks */]
    )
    useEffect(() => {
        const sub = eventBus.listenQueueing(createEditorEvent.match, (event: Action<EditorEvent>) => {
            handleEvent(event.payload)
        })
        return () => sub.unsubscribe()
    }, [handleEvent])

    const documentChangeShortTimeoutRef = useRef(0)
    const documentChangeLongTimeoutRef = useRef(0)
    const onDocumentChange = (document: Document, editorId: EditorId) => {
        didGenerateRef.current -= 1
        if (meta.id !== editorId) {
            logWarning('documentChange handler mismatch')
            return
        }
        clearTimeout(documentChangeShortTimeoutRef.current)
        clearTimeout(documentChangeLongTimeoutRef.current)
        documentChangeShortTimeoutRef.current = setTimeout(() => {
            if (controlsRef.current)
                controlsRef.current.state = {
                    blocked: blockedRef.current,
                    canUndo: document.canPopHistory(),
                    canRedo: document.canDescendHistory(),
                    canRetry: document.canPopHistory() && didGenerateRef.current > 0,
                    branches: [...document.getDescendents()].map((id) => ({
                        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                        node: document.getDescendent(id)!,
                        preferred: document.getHistoryNode().route === id,
                    })),
                }
        }, 50) as unknown as number
        documentChangeLongTimeoutRef.current = setTimeout(() => {
            //updateLinks(document)
            relayDocumentChange()
        }, 500) as unknown as number
    }

    const logout = useLogout()

    //This should probably go to its own file.
    const onResponseRef = useRef(null) as MutableRefObject<null | ((response: string) => void)>
    const onRequest: RequestWrapper = useRecoilCallback(
        ({ snapshot }) =>
            async (
                story,
                request,
                startIndex,
                endIndex,
                onGenerationComplete,
                onGenerationError,
                onGenerationUpdate,
                context
            ) => {
                const settings = (await snapshot.getPromise(SessionValue('settings'))) as UserSettings
                let combinedResponse = ''
                let currentResponse = ''
                let currentIndex = 0
                const combinedTokens: number[][] = []
                const logprobsArr: LogProbs[][] = []
                const receivedTokens: { token: string; final: boolean }[] = []
                let finalAdded = false
                const shouldComment = story.getStoryText().length - endIndex < 100
                await new Promise((resolve, reject) => {
                    if (getUserSetting(settings, 'streamResponses')) {
                        request.requestStream(
                            async (token, index, final, tokenArr, logProbs) => {
                                if (finalAdded) return false
                                combinedTokens[index] = tokenArr
                                if (logProbs) logprobsArr[index] = logProbs
                                receivedTokens[index] = { token, final }
                                for (let i = currentIndex; i < receivedTokens.length; i++) {
                                    const element = receivedTokens[i]
                                    if (element !== undefined) {
                                        currentIndex++
                                        const replaced = element.token.replace(/\r/g, '')
                                        currentResponse += replaced
                                        finalAdded = element.final
                                    } else {
                                        break
                                    }
                                }
                                if (
                                    index === 0 &&
                                    context.spacesTrimmed > 0 &&
                                    currentResponse.startsWith(' ')
                                ) {
                                    currentResponse = currentResponse.replace(/^\s+/g, '')
                                }
                                if (final && context.preContextText[endIndex - 1] === ' ') {
                                    currentResponse += ' '
                                }
                                if (currentResponse !== '') {
                                    onResponseRef.current?.call(onResponseRef, currentResponse)
                                    combinedResponse += currentResponse
                                    currentResponse = ''
                                }
                                if (finalAdded) {
                                    setTimeout(() => {
                                        onGenerationComplete(
                                            combinedResponse,
                                            combinedTokens.flat(),
                                            logprobsArr.flat(),
                                            shouldComment
                                        )
                                        resolve(null)
                                    }, 20)
                                    return false
                                }
                                const resume = await onGenerationUpdate()
                                return resume
                            },
                            (error) => {
                                onGenerationError(error)
                                reject(null)
                            }
                        )
                    } else {
                        request
                            .request()
                            .then((response) => {
                                if (!response?.text) {
                                    onGenerationError({ status: response.status, message: response.error })
                                    reject(null)
                                    return
                                }
                                currentResponse = response.text.replace(/\r/g, '')
                                currentResponse = trimBrokenUnicode(currentResponse)
                                if (context.spacesTrimmed > 0 && currentResponse.startsWith(' ')) {
                                    currentResponse = currentResponse.replace(/^\s+/g, '')
                                }
                                if (context.preContextText[endIndex - 1] === ' ') {
                                    currentResponse += ' '
                                }
                                onResponseRef.current?.call(onResponseRef, currentResponse)
                                onGenerationComplete(
                                    currentResponse,
                                    response.tokens ?? [],
                                    response.logprobs,
                                    shouldComment
                                )
                                resolve(null)
                            })
                            .catch((error) => {
                                onGenerationError(error)
                                reject(null)
                            })
                    }
                })
            },
        []
    )
    const onError = useRecoilCallback(
        ({ set }) =>
            (error: GenerateError) => {
                blockedRef.current = false
                switch (error.type) {
                    case GenerateErrorType.generic: {
                        toast(error.message)
                        break
                    }
                    case GenerateErrorType.requestFailed: {
                        toast(error.message)
                        break
                    }
                    case GenerateErrorType.modelUnavailable: {
                        toast(
                            `Selected model ${error.model} is not available at your current subscription tier. 
                            The selected model has been changed to ${error.changedTo}. 
                            Your settings have not been changed and may need to be adjusted.`
                        )
                        break
                    }
                    case GenerateErrorType.contextSetup: {
                        toast(
                            `Bottom of context is not story text. 
                            This is likely caused by altered context settings and will cause 
                            generations to be disconnected from the current narrative.`
                        )
                        break
                    }
                    case GenerateErrorType.trialUsed: {
                        set(TrialUsedModal, true)
                        break
                    }
                    case GenerateErrorType.freeLimitReached: {
                        set(IPLimitModal, true)
                        break
                    }
                    case GenerateErrorType.noSubscription: {
                        set(SubscriptionDialogOpen, { open: true, blocked: false })
                        break
                    }
                    case GenerateErrorType.unauthorized: {
                        logout()
                        break
                    }
                }
            },
        []
    )
    const generate = useGenerate({
        onRequest,
        onError,
    })
    const onRequestGeneration = useRecoilCallback(
        ({ snapshot }) =>
            async (onResponse: (text: string) => void, text: string, start?: number, end?: number) => {
                if (blockedRef.current) return

                blockedRef.current = true
                if (story.document) updateState(story.document)

                onResponseRef.current = onResponse

                const inputMode = await snapshot.getPromise(SelectedInputMode)
                const inputModes = await snapshot.getPromise(InputModes)
                const eventHandler = new EventHandler(story, meta, inputMode, inputModes)

                logDebug('generation request', text, text.length, start, end)
                await generate(text, meta, story, eventHandler, start, end).finally(() => {
                    blockedRef.current = false
                    didGenerateRef.current = 2
                    story.didGenerate = true
                    if (story.document) updateState(story.document)
                })
            },
        [meta, story]
    )

    if (!story.document) return <BodyLarge400>No Document</BodyLarge400>
    if (!(story.document instanceof Document))
        return <BodyLarge400>Document Incorrectly Serialized</BodyLarge400>
    return (
        <ContainerFrame>
            <EditorControls ref={controlsRef} editorRef={editorRef} />
            <Editor
                ref={editorRef}
                editorId={meta.id}
                document={story.document}
                onDocumentChange={onDocumentChange}
                onRequestGeneration={onRequestGeneration}
            />
        </ContainerFrame>
    )
}

const ContainerFrame = styled.View`
    height: 100%;
    width: 100%;
    padding: 20px;
`
