# Timeout on system calls like `fnctl.lock` by using alarm signals.
import signal

from decorator import contextmanager

try:
    # Posix based file locking (Linux, Ubuntu, MacOS, etc.)
    #   Only allows locking on writable files, might cause
    #   strange results for reading.
    import fcntl, os

    def lock_file(f):
        if f.writable(): fcntl.lockf(f, fcntl.LOCK_EX)

    def unlock_file(f):
        if f.writable(): fcntl.lockf(f, fcntl.LOCK_UN)
except ModuleNotFoundError:
    # Windows file locking
    import msvcrt, os

    def file_size(f):
        return os.path.getsize(os.path.realpath(f.name))

    def lock_file(f):
        msvcrt.locking(f.fileno(), msvcrt.LK_RLCK, file_size(f))

    def unlock_file(f):
        msvcrt.locking(f.fileno(), msvcrt.LK_UNLCK, file_size(f))


@contextmanager
def timeout(seconds):
    def timeout_handler(signum, frame):
        raise InterruptedError

    original_handler = signal.signal(signal.SIGALRM, timeout_handler)

    try:
        signal.alarm(seconds)
        yield
    finally:
        signal.alarm(0)
        signal.signal(signal.SIGALRM, original_handler)


# Class for ensuring that all file operations are atomic, treat
# initialization like a standard call to 'open' that happens to be atomic.
# This file opener *must* be used in a "with" block.
class AtomicLockFileWithTimeout:
    # Open the file with arguments provided by user. Then acquire
    # a lock on that file object (WARNING: Advisory locking).
    def __init__(self, path, wait=3, *args, **kwargs):
        with timeout(wait):
            # Open the file and acquire a lock on the file before operating
            self.path = path
            self.file = open(path, *args, **kwargs)
            print(f"Trying to lock {self.path}")
            # Lock the opened file
            try:
                lock_file(self.file)
                print(f"Got lock on {self.path}")
            except InterruptedError:
                print(f"{path} lock attempt timed out")
                self.file = None

    # Return the opened file object (knowing a lock has been obtained).
    # This should be checked if `None`, as `None` means that a lock has NOT
    # been obtained.
    def __enter__(self, *args, **kwargs):
        return self.file

    # Unlock the file and close the file object.
    def __exit__(self, exc_type=None, exc_value=None, traceback=None):
        if self.file is not None:
            # Flush to make sure all buffered contents are written to file.
            self.file.flush()
            os.fsync(self.file.fileno())
            # Release the lock on the file.
            try:
                os.remove(self.path)
            except FileNotFoundError:
                pass
            unlock_file(self.file)
            self.file.close()
        # Handle exceptions that may have come up during execution, by
        # default any exceptions are raised to the user.
        return exc_type is None